# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import socket
from decimal import Decimal

from common.utils import gen_hex_uuid
from travel.rasp.library.python.common23.date import environment
from common.utils.exceptions import SimpleUnicodeException
from common.workflow.registry import run_process
from travel.rasp.train_api.train_partners.base.get_order_info import get_order_info
from travel.rasp.train_api.train_partners.base.refund import TicketRefundResult
from travel.rasp.train_api.train_partners.base.update_order import update_order_info
from travel.rasp.train_api.train_purchase.core.enums import TrainPartner
from travel.rasp.train_api.train_purchase.core.models import RefundStatus, RefundUserInfo, TrainRefund
from travel.rasp.train_api.train_purchase.workflow.ticket_refund import TICKET_REFUND_PROCESS

log = logging.getLogger(__name__)


class ExternalRefundError(SimpleUnicodeException):
    pass


class ExternalRefundCriticalError(ExternalRefundError):
    pass


def create_external_train_refund_and_run_it(order, refund_yandex_fee):
    if order.partner != TrainPartner.IM:
        raise ExternalRefundCriticalError('Не обрабатываются кассовые возвраты по заказам, '
                                          'сделанным через {}'.format(order.partner))
    try:
        update_order_info(order)
        order_info = get_order_info(order)
        refunded_blank_ids = _get_refunded_or_inprocess_blank_ids(order)
        refund_by_blank_ids = _get_refund_by_blank_ids(order_info, refunded_blank_ids)
        refunded_insurance_ids = _get_refunded_or_inprocess_insurance_ids(order, refunded_blank_ids)
        insurance_ids = [i.operation_id for i in order_info.insurances
                         if i.refund_amount and i.operation_id not in refunded_insurance_ids] or None

        if not refund_by_blank_ids and not insurance_ids:
            raise ExternalRefundError('Для данного кассового возврата уже созданы наши возвраты')

        refund_uuid = gen_hex_uuid()
        _update_order(order, refund_by_blank_ids, refund_uuid, insurance_ids, refund_yandex_fee)
        refund = TrainRefund.objects.create(
            uuid=refund_uuid,
            order_uid=order.uid,
            is_active=True,
            status=RefundStatus.NEW,
            blank_ids=list(refund_by_blank_ids) if refund_by_blank_ids else None,
            insurance_ids=insurance_ids,
            user_info=RefundUserInfo(ip='127.0.0.1', uid=socket.gethostname(), region_id=213),
            is_external=True,
            created_at=environment.now_utc(),
        )

        run_process.apply_async([TICKET_REFUND_PROCESS, str(refund.id), {'order_uid': order.uid}])
    except Exception:
        log.exception('Error in OfficeRefund')
        raise ExternalRefundCriticalError('Error in OfficeRefund')


def _get_refund_by_blank_ids(order_info, refunded_blank_ids):
    refund_by_blank_id = {}
    for ticket_info in order_info.tickets:
        if not (ticket_info.is_external and ticket_info.refund_operation_id):
            continue
        if ticket_info.blank_id in refunded_blank_ids:
            continue
        refund_by_blank_id[ticket_info.blank_id] = TicketRefundResult(
            amount=ticket_info.refund_amount,
            refund_blank_id=ticket_info.refund_blank_id,
            refund_operation_id=ticket_info.refund_operation_id,
            rzhd_status=ticket_info.rzhd_status,
        )

    return refund_by_blank_id


def _get_refunded_or_inprocess_blank_ids(order):
    refunded_blank_ids = set()
    for refund in order.iter_refunds():
        if refund.status != RefundStatus.FAILED:
            refunded_blank_ids.update(refund.blank_ids)

    return refunded_blank_ids


def _get_refunded_or_inprocess_insurance_ids(order, refunded_blank_ids):
    # Сейчас при создании обычного возврата refund.insurance_ids не заполняются, поэтому вычисляем по бланкам.
    refunded_insurance_ids = set()
    for passenger in order.passengers:
        blank_ids = [t.blank_id for t in passenger.tickets]
        if all(b in refunded_blank_ids for b in blank_ids):
            if passenger.insurance and passenger.insurance.operation_id:
                refunded_insurance_ids.add(passenger.insurance.operation_id)

    # Дополнительно надо учесть автовозвраты страховок и кассовые возвраты.
    for refund in order.iter_refunds():
        if refund.status != RefundStatus.FAILED and refund.insurance_ids:
            refunded_insurance_ids.update(refund.insurance_ids)

    return refunded_insurance_ids


def _update_order(order, refund_by_blank_ids, refund_uuid, insurance_ids, refund_yandex_fee):
    update_spec = {}

    for ticket, ticket_lookup_name in order.iter_ticket_to_lookup_name():
        if ticket.blank_id not in refund_by_blank_ids:
            continue
        ticket_refund_result = refund_by_blank_ids[ticket.blank_id]
        # TODO: здесь мы уже знаем id бланка возврата ticket_refund_result.refund_blank_id
        # Можно вписать его в заказ и использовать для отправки письма
        ticket_prefix = 'set__{}__'.format(ticket_lookup_name)
        refund_prefix = ticket_prefix + 'refund__'

        if ticket.payment.amount > 0:
            update_spec.update({
                refund_prefix + 'amount': ticket_refund_result.amount,
                # Возврат сбора мог остаться с запроса суммы к возврату, поэтому расчитываем и перезаписываем его.
                refund_prefix + 'refund_yandex_fee_amount':
                    ticket.payment.refundable_yandex_fee_amount if refund_yandex_fee else Decimal(0),
            })
        update_spec.update({
            refund_prefix + 'operation_id': ticket_refund_result.refund_operation_id,
            ticket_prefix + 'rzhd_status': ticket_refund_result.rzhd_status,
        })

    if insurance_ids:
        for i, passenger in enumerate(order.passengers):
            if (passenger.insurance and passenger.insurance.operation_id in insurance_ids
                    and passenger.insurance.trust_order_id):
                update_spec['set__passengers__{}__insurance__refund_uuid'.format(i)] = refund_uuid

    order.update(**update_spec)
