# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import timedelta, datetime
from decimal import Decimal

from enum import Enum
from six import text_type

from common.utils.date import UTC_TZ
from travel.rasp.library.python.common23.date.environment import now_utc
from common.utils.try_hard import try_hard
from common.workflow import registry
from common.workflow.registry import get_process
from travel.rasp.train_api.train_partners.base import RzhdStatus, PartnerError
from travel.rasp.train_api.train_partners.base.get_order_info import get_order_info
from travel.rasp.train_api.train_partners.base.refund_amount import get_refund_amount
from travel.rasp.train_api.train_partners.base.update_order import update_order_info
from travel.rasp.train_api.train_purchase.core.enums import OrderWarningCode

log = logging.getLogger(__name__)

TICKETS_EXPIRE_ER_WARNING_TIME = timedelta(minutes=15)
INSURANCE_WARNING_ACTUAL_TIME = timedelta(minutes=3)
MIN_DATETIME_UTC = UTC_TZ.localize(datetime.min)
MAX_DATETIME_UTC = UTC_TZ.localize(datetime.max)
YANDEX_FEE_REFUND_TIME = timedelta(hours=24)


def update_refund_amount(order, blank_ids):
    """
    :type order: common.apps.train_order.models.TrainOrder
    """
    try:
        doc_id = _get_order_info(order).passengers[0].doc_id
    except IndexError:
        raise Exception('Can not fetch doc id')

    blanks = get_blanks_with_refund_amount(order, doc_id, blank_ids)
    order_update = make_ticket_refund_update(order, blanks)
    if order_update:
        order.modify(**order_update)


@try_hard()
def _get_order_info(order):
    return get_order_info(order)


def should_refund_yandex_fee(order, refund_started_at=None):
    if not order.finished_at:
        return False

    if not refund_started_at:
        refund_started_at = now_utc()

    time_since_purchase = refund_started_at - order.finished_at
    return time_since_purchase < YANDEX_FEE_REFUND_TIME


def make_ticket_refund_update(order, blanks):
    order_update = {}
    blanks_by_id = {blank.id: blank for blank in blanks}
    refund_yandex_fee = should_refund_yandex_fee(order)

    for ticket, ticket_lookup_name in order.iter_ticket_to_lookup_name():
        if ticket.blank_id not in blanks_by_id:
            continue

        if ticket.payment.amount == 0:
            continue

        blank = blanks_by_id[ticket.blank_id]
        prefix = 'set__{}__refund__'.format(ticket_lookup_name)
        order_update.update({
            prefix + 'amount': blank.amount,
            prefix + 'tariff_vat__rate': blank.tariff_vat.rate,
            prefix + 'tariff_vat__amount': blank.tariff_vat.amount,
            prefix + 'service_vat__rate': blank.service_vat.rate,
            prefix + 'service_vat__amount': blank.service_vat.amount,
            prefix + 'commission_fee_vat__rate': blank.commission_fee_vat.rate,
            prefix + 'commission_fee_vat__amount': blank.commission_fee_vat.amount,
            prefix + 'refund_commission_fee_vat__rate': blank.refund_commission_fee_vat.rate,
            prefix + 'refund_commission_fee_vat__amount': blank.refund_commission_fee_vat.amount,
            # Возврат сбора мог остаться с прошлого запроса суммы к возврату, поэтому расчитываем и перезаписываем его.
            prefix + 'refund_yandex_fee_amount': (ticket.payment.refundable_yandex_fee_amount
                                                  if refund_yandex_fee else Decimal(0))
        })

    return order_update


def iter_refundable_tickets(order):
    return (
        t for t in order.iter_tickets()
        if t.rzhd_status is not None and RzhdStatus(t.rzhd_status).is_refundable()
    )


def get_blanks_with_refund_amount(order, doc_id, blank_ids):
    all_blank_ids = {t.blank_id for t in order.iter_tickets()}
    if len(blank_ids) == len(all_blank_ids):
        blanks = get_refund_amount(order, doc_id)
    else:
        blanks = []
        for blank_id in blank_ids:
            blanks.extend(get_refund_amount(order, doc_id, blank_id))

    return blanks


class UpdateTicketStatusMode(Enum):
    SIMPLE = 'simple'
    UPDATE_FROM_EXPRESS = 'update_from_express'
    TRY_UPDATE_FROM_EXPRESS = 'tru_update_from_express'


def update_tickets_statuses(order, mode=UpdateTicketStatusMode.SIMPLE, update_order=True, set_personal_data=False,
                            set_order_warnings=False, first_actual_warning_only=False):
    """
    :type order: common.apps.train_order.models.TrainOrder
    :param mode: train_api.train_purchase.utils.order.UpdateTicketStatusMode
    :param update_order: bool
    :param set_personal_data: bool
    :param set_order_warnings: bool
    :param first_actual_warning_only: bool
    update_order_info ходит в express поэтому поставщики его не любят
    """
    if mode in [UpdateTicketStatusMode.UPDATE_FROM_EXPRESS, UpdateTicketStatusMode.TRY_UPDATE_FROM_EXPRESS]:
        try:
            update_order_info(order)
        except PartnerError as err:
            if err.is_update_from_express_error() and mode == UpdateTicketStatusMode.TRY_UPDATE_FROM_EXPRESS:
                log.warning('Не удалось обновить заказ в АСУ Экспресс')
            else:
                raise

    if update_order or set_personal_data or set_order_warnings:
        order_info = get_order_info(order)
        if update_order:
            tickets = order_info.tickets
            order_update = make_ticket_statuses_update(order, tickets)
            if order_update:
                order.modify(**order_update)
        if set_personal_data:
            passengers_info = {p.customer_id: p for p in order_info.passengers}
            for p in order.passengers:
                if p.customer_id in passengers_info:
                    p.doc_id = passengers_info[p.customer_id].doc_id
                    p.birth_date = passengers_info[p.customer_id].birth_date
        if set_order_warnings:
            order.warnings = get_order_warnings(order, order_info)
            if first_actual_warning_only:
                order.warnings = get_first_actual_warning(order.warnings)


def get_order_warnings(order, order_info):
    """
    Функция возвращает список текущих и будущих предупреждений, упорядоченный по важности
    :param order:
    :param order_info:
    :return: список предупреждений
    """
    warnings = []
    tickets_returned = all(t.rzhd_status == RzhdStatus.REFUNDED for t in order_info.tickets)
    if tickets_returned:
        return warnings
    tickets_taken_away = all(t.rzhd_status == RzhdStatus.STRICT_BOARDING_PASS for t in order_info.tickets)
    if tickets_taken_away:
        warnings.append({
            'code': OrderWarningCode.TICKETS_TAKEN_AWAY.value,
        })
        return warnings
    now = now_utc(aware=True)
    if order.finished_at and order.insurance_auto_return_uuid:
        finished_at = UTC_TZ.localize(order.finished_at)
        insurance_warning_actual_till = finished_at + INSURANCE_WARNING_ACTUAL_TIME
        if now < insurance_warning_actual_till:
            warnings.append({
                'from': finished_at,
                'to': insurance_warning_actual_till,
                'code': OrderWarningCode.INSURANCE_AUTO_RETURN.value,
            })
    # Когда в заказе есть билеты с разным статусом ЭР (например, для одного билета ЭР есть, для другого нет),
    # нужно возвращать предупреждения для билета, у которого нет ЭР (для этого билета времени на возврат больше).
    if order_info.expire_set_er and not any(t.rzhd_status == RzhdStatus.NO_REMOTE_CHECK_IN for t in order_info.tickets):
        expire_dt = order_info.expire_set_er.astimezone(UTC_TZ)
        start_station_departure = None
        if order.route_info and order.route_info.start_station and order.route_info.start_station.departure:
            start_station_departure = UTC_TZ.localize(order.route_info.start_station.departure)
        if expire_dt and start_station_departure and expire_dt < start_station_departure:
            if now < expire_dt:
                warnings.append({
                    'from': expire_dt - TICKETS_EXPIRE_ER_WARNING_TIME,
                    'to': expire_dt,
                    'code': OrderWarningCode.ELECTRONIC_REGISTRATION_ALMOST_EXPIRED.value,
                })
            if now < start_station_departure:
                warnings.append({
                    'from': expire_dt,
                    'code': OrderWarningCode.ELECTRONIC_REGISTRATION_EXPIRED.value,
                    'to': start_station_departure,
                })
            warnings.append({
                'from': start_station_departure,
                'code': OrderWarningCode.TRAIN_LEFT_START_STATION.value,
            })
        else:
            if now < expire_dt:
                warnings.append({
                    'from': expire_dt - TICKETS_EXPIRE_ER_WARNING_TIME,
                    'to': expire_dt,
                    'code': OrderWarningCode.ELECTRONIC_REGISTRATION_ALMOST_EXPIRED.value,
                })
            warnings.append({
                'from': expire_dt,
                'code': OrderWarningCode.ELECTRONIC_REGISTRATION_EXPIRED.value,
            })
    else:
        departure = UTC_TZ.localize(order.departure)
        if now < departure:
            warnings.append({
                'from': departure - TICKETS_EXPIRE_ER_WARNING_TIME,
                'to': departure,
                'code': OrderWarningCode.TRAIN_ALMOST_LEFT_DEPARTURE_STATION.value,
            })
        warnings.append({
            'from': departure,
            'code': OrderWarningCode.TRAIN_LEFT_DEPARTURE_STATION.value,
        })
    return warnings


def get_first_actual_warning(warnings):
    now = now_utc(aware=True)
    for warning in warnings:
        if warning.get('from', MIN_DATETIME_UTC) <= now < warning.get('to', MAX_DATETIME_UTC):
            return [warning]
    return []


def make_ticket_statuses_update(order, ticket_infos):
    blank_id_to_ticket_info = {text_type(t.blank_id): t for t in ticket_infos}
    order_update = {}
    for ticket, ticket_lookup_name in order.iter_ticket_to_lookup_name():
        blank_id = ticket.blank_id
        try:
            ticket_info = blank_id_to_ticket_info[blank_id]
        except KeyError:
            log.error('Не нашли статус бланка %s в UpdateOrderInfo', blank_id)
        else:
            order_update['set__{}__rzhd_status'.format(ticket_lookup_name)] = ticket_info.rzhd_status
            order_update['set__{}__pending'.format(ticket_lookup_name)] = ticket_info.pending

            has_ticket_refund_operation_id = ticket.refund and ticket.refund.operation_id
            if ticket_info.refund_operation_id and not has_ticket_refund_operation_id:
                order_update['set__{}__refund__operation_id'.format(ticket_lookup_name)] = \
                    ticket_info.refund_operation_id

    return order_update


def make_order_status_update(order, order_info):
    partner_data_lookup_name = order.current_partner_data_lookup_name
    order_update = {
        'set__{}__operation_status'.format(partner_data_lookup_name): order_info.status,
        'set__{}__order_num'.format(partner_data_lookup_name): order_info.order_num
    }
    if not order.current_partner_data.expire_set_er and order_info.expire_set_er:
        order_update['set__{}__expire_set_er'.format(partner_data_lookup_name)] = order_info.expire_set_er
    return order_update


def send_event_to_order(order, event_name, params=None, allow_send_to_empty_process=False):
    from travel.rasp.train_api.train_purchase.workflow.booking import TRAIN_BOOKING_PROCESS

    process = get_process(TRAIN_BOOKING_PROCESS, order)
    process.send_external_event(event_name, params, allow_send_to_empty_process=allow_send_to_empty_process)
    registry.run_process.apply_async([TRAIN_BOOKING_PROCESS, str(order.id), {'order_uid': order.uid}])


def send_event_to_refund(refund, event_name, params=None):
    from travel.rasp.train_api.train_purchase.workflow.ticket_refund import TICKET_REFUND_PROCESS

    process = get_process(TICKET_REFUND_PROCESS, refund)
    process.send_external_event(event_name, params)
    registry.run_process.apply_async([TICKET_REFUND_PROCESS, str(refund.id), {'order_uid': refund.order_uid}])


def send_event_to_payment(payment, event_name, params=None, allow_send_to_empty_process=False):
    from travel.rasp.train_api.train_purchase.workflow.payment import PAYMENT_PROCESS

    process = get_process(PAYMENT_PROCESS, payment)
    process.send_external_event(event_name, params, allow_send_to_empty_process)
    registry.run_process.apply_async(
        [PAYMENT_PROCESS, str(payment.id), {'order_uid': payment.order_uid, 'payment': payment.uid}]
    )
