# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import timedelta
from decimal import Decimal

from dateutil.relativedelta import relativedelta
from django.utils.encoding import force_text

from common.data_api.billing.trust_client import TrustClient, TrustPaymentStatuses
from common.dynamic_settings.default import conf
from common.models.geo import Station
from common.utils.date import UTC_TZ
from travel.rasp.library.python.common23.date.environment import now_aware, now_utc
from common.utils.railway import get_railway_tz_by_point
from travel.rasp.train_api.helpers.error import build_error_info
from travel.rasp.train_api.train_partners import get_partner_api
from travel.rasp.train_api.train_partners.base import PartnerError
from travel.rasp.train_api.train_partners.base.confirm_ticket import cancel_order
from travel.rasp.train_api.train_partners.base.get_order_info import get_order_info
from travel.rasp.train_api.train_partners.base.ticket_blank import BlankFormat, download_ticket_blank
from travel.rasp.train_api.train_purchase.core.enums import OrderStatus, RebookingStatus, TravelOrderStatus
from travel.rasp.train_api.train_purchase.core.models import (
    PartnerData, Passenger, Ticket, TicketPayment, TrainOrder, RefundBlank, Payment, RebookingInfo,
    PassengerRebookingInfo, PassengerTariffInfo, OrderRouteInfo, StationInfo
)
from travel.rasp.train_api.train_purchase.core.utils import hash_birth_date, hash_doc_id
from travel.rasp.train_api.train_purchase.utils.fee_calculator import calculate_ticket_cost
from travel.rasp.train_api.train_purchase.utils.order import send_event_to_order
from travel.rasp.train_api.train_purchase.workflow.user_events import TrainBookingUserEvents


log = logging.getLogger(__name__)


IM_TIMEOUT_IN_SEC = 5
MIN_BOOKING_TIME_IN_SEC = 60


class TicketNotFoundError(Exception):
    pass


class RebookingError(Exception):
    pass


def reserve_tickets(order_data):
    partner_api = get_partner_api(order_data['partner'])
    reservation_manager = partner_api.ReservationManager.from_order_data(order_data)
    return reservation_manager.reserve()


def _calculate_boarding_age(order_data, birth_date):
    departure_date = order_data['departure'].astimezone(get_railway_tz_by_point(order_data['station_from'])).date()
    return _calculate_boarding_age_by_dates(departure_date, birth_date)


def _calculate_boarding_age_by_dates(departure_date, birth_date):
    return relativedelta(departure_date, birth_date).years


def _make_ticket_payment(order_data, reserved_ticket, contract, yandex_uid=None):
    """
    :type order_data: dict
    :type reserved_ticket: train_api.train_partners.base.reserve_tickets.ReserveResponseTicket
    :type contract: common.apps.train_order.models.ClientContract
    :type experiment: Bool
    :rtype: TicketPayment
    """
    amount = reserved_ticket.amount
    if amount:
        partner_fee = contract.partner_commission_sum
        partner_refund_fee = contract.partner_commission_sum2
        service_price = reserved_ticket.service_price or Decimal(0)

        ticket_cost = calculate_ticket_cost(
            contract=contract,
            coach_type=order_data['coach_type'].value,
            amount=amount,
            service_amount=service_price,
            yandex_uid=yandex_uid,
        )
        bedding_fee = ticket_cost.bedding_fee if order_data['bedding'] else Decimal(0)
        fee = ticket_cost.main_fee + bedding_fee
        fee_percent = ticket_cost.yandex_fee_percent
    else:
        service_price = Decimal(0)
        fee = fee_percent = partner_fee = partner_refund_fee = bedding_fee = Decimal(0)
    return TicketPayment(
        amount=amount,
        service_amount=service_price,
        service_fee=bedding_fee,
        fee=fee,
        fee_percent=fee_percent,
        fee_percent_range=Decimal(conf.TRAIN_PURCHASE_EXPERIMENTAL_DELTA_FEE),
        partner_fee=partner_fee,
        partner_refund_fee=partner_refund_fee,
        tariff_vat=reserved_ticket.tariff_vat and reserved_ticket.tariff_vat._asdict(),
        service_vat=reserved_ticket.service_vat and reserved_ticket.service_vat._asdict(),
        commission_fee_vat=reserved_ticket.commission_fee_vat and reserved_ticket.commission_fee_vat._asdict()
    )


def _make_passengers(order_data, reserved_tickets, contract):
    passenger_data_by_index = {i: passenger for i, passenger in enumerate(order_data['passengers'])}
    yandex_uid = order_data['user_info']['yandex_uid'] if order_data['price_exp_id'] else None
    for reserved_ticket in reserved_tickets:
        birth_date = reserved_ticket.passenger_birth_date
        document_number = reserved_ticket.passenger_document_number
        passenger = passenger_data_by_index[reserved_ticket.index]
        yield Passenger(
            birth_date_hash=hash_birth_date(birth_date),
            doc_id_hash=hash_doc_id(document_number),

            first_name=reserved_ticket.passenger_first_name,
            last_name=reserved_ticket.passenger_last_name,
            patronymic=reserved_ticket.passenger_patronymic,
            sex=reserved_ticket.passenger_gender,
            age=_calculate_boarding_age(order_data, birth_date),
            doc_type=reserved_ticket.passenger_document_type,
            citizenship_country_id=reserved_ticket.passenger_citizenship_country.id,
            customer_id=reserved_ticket.customer_id,
            phone=passenger['phone'],
            email=passenger['email'],

            rebooking_info=PassengerRebookingInfo(
                age_group=passenger['age_group'],
                loyalty_cards=passenger['loyalty_cards'],
                tariff_info=PassengerTariffInfo.to_dict(passenger['tariff_info'])
            ),

            # TODO заменить tickets на ticket
            tickets=[
                Ticket(
                    blank_id=reserved_ticket.blank_id,
                    carrier_inn=reserved_ticket.carrier_inn,
                    places=reserved_ticket.places,
                    places_type=reserved_ticket.places_type,
                    payment=_make_ticket_payment(order_data, reserved_ticket, contract, yandex_uid=yandex_uid),
                    tariff_info_code=reserved_ticket.tariff_info and reserved_ticket.tariff_info.code,
                    raw_tariff_title=reserved_ticket.raw_tariff_title,
                )
            ],
        )


def _make_partner_data(reservation_result):
    partner_data = PartnerData(
        im_order_id=reservation_result.im_order_id,
        operation_id=reservation_result.operation_id,
        special_notice=reservation_result.special_notice,
        time_notice=reservation_result.time_notice,
        station_from_title=reservation_result.departure_station_title,
        station_to_title=reservation_result.arrival_station_title,
        compartment_gender=reservation_result.compartment_gender,
        is_three_hours_reservation_available=reservation_result.is_three_hours_reservation_available,
        is_suburban=reservation_result.is_suburban,
        reservation_datetime=now_utc(),
        is_only_full_return_possible=reservation_result.is_only_full_return_possible,
    )
    return partner_data


def _make_rebooking_info(order_data, partner_data):
    enabled = conf.TRAIN_PURCHASE_REBOOKING_ENABLED and order_data['enable_rebooking']
    cycle_until = now_utc() + timedelta(minutes=conf.TRAIN_PURCHASE_RESERVATION_PARTNER_TIMEOUT)
    rebooking_info = RebookingInfo(
        enabled=enabled,
        cycle_until=cycle_until if enabled and not partner_data.is_three_hours_reservation_available else None,
        bedding=order_data['bedding'],
        electronic_registration=order_data['electronic_registration'],
        place_demands=order_data['place_demands'],
        service_class=order_data['service_class'],
        international_service_class=order_data['international_service_class'],
        places=order_data['places'],
        requirements=order_data['requirements'],
        additional_place_requirements=order_data['additional_place_requirements'],
        give_child_without_place=order_data['give_child_without_place'],
        is_cppk=order_data['is_cppk'],
    )
    return rebooking_info


def _calculate_reserved_to(reservation_result, rebooking_info):
    if rebooking_info and rebooking_info.cycle_until:
        total_cycle_reservation = now_utc() + timedelta(
            minutes=conf.TRAIN_PURCHASE_RESERVATION_MAX_CYCLES * conf.TRAIN_PURCHASE_RESERVATION_PARTNER_TIMEOUT
        )
        prolong_reservation = now_utc() + timedelta(minutes=conf.TRAIN_PURCHASE_PROLONG_RESERVATION_MINUTES)
        return min(total_cycle_reservation, prolong_reservation)

    return reservation_result.reserved_to.astimezone(UTC_TZ).replace(tzinfo=None)


def _get_station_info_by_code(express_code, departure_datetime=None):
    try:
        station = Station.get_by_code('express', express_code)
        station_info = StationInfo.create_from_station(station)
        station_info.departure = departure_datetime
        return station_info
    except Exception as e:
        log.warning('Cannot get station by code %s: %s', express_code, force_text(e))
        return None


def _fill_points_ids_and_names(order_data, partner_data):
    route_info = OrderRouteInfo()
    station_from = order_data['station_from']
    station_to = order_data['station_to']

    try:
        partner_api = get_partner_api(order_data['partner'])
        result = partner_api.get_route_info(
            train_number=order_data['train_number'],
            station_from=station_from,
            station_to=station_to,
            departure=order_data['departure'].astimezone(UTC_TZ).replace(tzinfo=None),
            partner_credential_id=order_data['partner_credential_id'],
        )
        route_info.start_station = _get_station_info_by_code(result.first_stop.station_express_code,
                                                             result.first_stop.departure_datetime)
        route_info.end_station = _get_station_info_by_code(result.last_stop.station_express_code)

        partner_data.start_station_title = result.first_stop.station_name
        partner_data.end_station_title = result.last_stop.station_name

    except Exception as e:
        log.warning('Error in obtaining start/end stations: %s', force_text(e))

    route_info.from_station = StationInfo.create_from_station(station_from)
    route_info.to_station = StationInfo.create_from_station(station_to)

    order_data.update({
        'route_info': route_info,
    })

    return order_data


ORDER_FIELDS_FROM_DATA = (
    'coach_type',
    'gender',
    'partner',
    'partner_credential_id',
    'price_exp_id',
    'route_info',
    'route_policy',
    'scheme_id',
    'source',
    'train_name',
    'train_number',
    'train_ticket_number',
    'two_storey',
    'user_info',
)


def create_order(order_uid, order_data, reservation_result, contract):
    Payment.objects.create(order_uid=order_uid)
    reserved_tickets = reservation_result.tickets
    order_data['user_info']['reversed_phone'] = order_data['user_info']['phone'][::-1]
    partner_data = _make_partner_data(reservation_result)
    insurance_enabled = conf.TRAIN_PURCHASE_INSURANCE_ENABLED
    order_data = _fill_points_ids_and_names(order_data, partner_data)
    rebooking_info = _make_rebooking_info(order_data, partner_data)
    order = TrainOrder.objects.create(
        uid=order_uid,
        status=OrderStatus.RESERVED,
        travel_status=TravelOrderStatus.RESERVED,
        passengers=list(_make_passengers(order_data, reserved_tickets, contract)),
        station_from_id=order_data['station_from'].id,
        station_to_id=order_data['station_to'].id,
        arrival=order_data['arrival'].astimezone(UTC_TZ).replace(tzinfo=None),
        departure=order_data['departure'].astimezone(UTC_TZ).replace(tzinfo=None),
        displayed_coach_owner=order_data.get('coach_owner'),
        coach_owner=reservation_result.coach_owner,
        coach_number=reservation_result.coach_number,
        reserved_to=_calculate_reserved_to(reservation_result, rebooking_info),
        partner_data_history=[partner_data],
        orders_created=[
            a['data'] for a in reversed(order_data['order_history']) if a['type'] == 'SET_ORDER_CREATED'
        ] if order_data.get('order_history') else None,
        rebooking_info=rebooking_info,
        insurance_enabled=insurance_enabled,
        process={'suspended': True} if insurance_enabled else {},
        **{field: order_data[field] for field in ORDER_FIELDS_FROM_DATA if field in order_data}
    )
    personal_data = [{
        'birth_date': reserved_ticket.passenger_birth_date,
        'doc_id': reserved_ticket.passenger_document_number
    } for reserved_ticket in reserved_tickets]
    return order, personal_data


def apply_personal_data(order, personal_data):
    for passenger, data in zip(order.passengers, personal_data):
        for field_name, value in data.items():
            setattr(passenger, field_name, value)


def download_refund_blank(order, ticket):
    refund = ticket.refund
    blank = refund.get_blank()
    if not blank:
        blank_content = download_ticket_blank(order, refund.operation_id, BlankFormat.PDF)
        blank = RefundBlank.objects.create(content=blank_content)
        for order_ticket, lookup_name in order.iter_ticket_to_lookup_name():
            if order_ticket == ticket:
                ticket_lookup_name = lookup_name
                break
        else:
            raise TicketNotFoundError('Ticket with blank {} not found in Order {}'.format(ticket.blank_id, order.uid))
        order.modify(**{'set__{}__refund__blank_id'.format(ticket_lookup_name): blank.pk})
    return blank


def rebooking(order):
    try:
        if order.rebooking_info is None:
            raise RebookingError('Order {} has not rebooking_info'.format(order.uid))

        if order.status == OrderStatus.DONE or getattr(order.process, 'state', None) == 'unhandled_exception_state'\
                or getattr(order.current_billing_payment.process, 'state', None) == 'unhandled_exception_state':
            raise RebookingError('Cannot rebook due state of order {}'.format(order.uid))

        if not TrainOrder.objects.filter(
            uid=order.uid, rebooking_info__status__ne=RebookingStatus.IN_PROCESS
        ).modify(**{'set__rebooking_info__status': RebookingStatus.IN_PROCESS}):
            raise RebookingError('Order {} already moved in process'.format(order.uid))

        payment_status = TrustClient().get_payment_status(order.current_billing_payment.purchase_token)
        if payment_status != TrustPaymentStatuses.AUTHORIZED:
            raise RebookingError('Order {}, payment_status is "{}" instead "authorized"'.format(
                order.uid, payment_status)
            )

        order_info = get_order_info(order, timeout=IM_TIMEOUT_IN_SEC)
        remaining_booking_time_in_sec = ((order_info.reserved_to or order.reserved_to) - now_aware()).total_seconds()
        if remaining_booking_time_in_sec >= MIN_BOOKING_TIME_IN_SEC:
            # не нужно перебронирование
            order.update(**{'set__rebooking_info__status': RebookingStatus.SKIPPED})
        else:
            try:
                success, update_spec, mismatch = re_reserve_tickets_and_check(
                    order,
                    order_info,
                    cancel_required=remaining_booking_time_in_sec > 0,
                    timeout=IM_TIMEOUT_IN_SEC,
                )
                order.update(**update_spec)
                if not success:
                    raise RebookingError(mismatch)

            except Exception as error:
                try:
                    update_spec = {
                        'set__error': build_error_info(error),
                        'set__{}__is_order_cancelled'.format(order.current_partner_data_lookup_name): True,
                    }
                    cancel_order(order, timeout=IM_TIMEOUT_IN_SEC)
                    order.update(**update_spec)
                except Exception:
                    log.exception('Error while trying to cancel order {}'.format(order.uid))
                raise error

            order.update(**{'set__rebooking_info__status': RebookingStatus.DONE})

        send_event_to_order(order, TrainBookingUserEvents.PAID)

    except Exception as error:
        log.exception('Error in rebooking order {}'.format(order.uid))
        order.update(**{'set__rebooking_info__status': RebookingStatus.FAILED})
        raise error


def re_reserve_tickets_and_check(order, order_info, cancel_required=False, timeout=None):
    if not getattr(order, 'station_from', None) or not getattr(order, 'station_to', None):
        TrainOrder.fetch_stations([order])

    if cancel_required:
        try:
            cancel_order(order, timeout=timeout)
        except PartnerError as error:
            if not error.is_update_from_express_error():
                raise

    partner_api = get_partner_api(order.partner)
    reservation_manager = partner_api.ReservationManager.from_order(order, order_info)
    reservation_result = reservation_manager.reserve(timeout=timeout)

    partner_data = _make_partner_data(reservation_result).to_mongo().to_dict()
    update_spec = {'push__partner_data_history': partner_data}

    reserved_tickets_by_index = {ticket.index: ticket for ticket in reservation_result.tickets}
    success = True
    mismatch = None
    for i, passenger in enumerate(order.passengers):
        reserved_ticket = reserved_tickets_by_index[i]

        old_places = set(passenger.tickets[0].places or [])
        new_places = set(reserved_ticket.places or [])
        if old_places != new_places:
            success = False
            mismatch = 'old places "{}" != new places "{}"'.format(old_places, new_places)

        old_amount = passenger.tickets[0].payment.amount
        new_amount = reserved_ticket.amount
        if old_amount != new_amount:
            success = False
            mismatch = 'old amount "{}" != new amount "{}"'.format(passenger, new_places)

        update_spec.update({
            'set__passengers__{}__tickets__0__blank_id'.format(i): reserved_ticket.blank_id,
            'set__passengers__{}__customer_id'.format(i): reserved_ticket.customer_id,
        })

    return success, update_spec, mismatch
