# coding: utf8

from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from decimal import Decimal

from django.utils import translation

from common.data_api.billing.trust_client import train_order_skip_trust
from common.email_sender import guaranteed_send_email, AttachmentInfo
from common.settings.configuration import Configuration
from common.settings.utils import get_setting
from travel.rasp.library.python.common23.date.environment import now_aware
from travel.rasp.train_api.train_partners.base.ticket_blank import BlankFormat
from travel.rasp.train_api.train_purchase.core.models import TrainOrder, TrainRefund
from travel.rasp.train_api.train_purchase.core.utils import format_money
from travel.rasp.train_api.train_purchase.utils.billing import download_refund_receipt
from travel.rasp.train_api.train_purchase.utils.email_sender_callback import report_error_email
from travel.rasp.train_api.train_purchase.utils.order_tickets import download_refund_blank
from travel.rasp.train_api.train_purchase.utils.tickets_email import make_mail_args_for_blanks, can_skip_trust_check

log = logging.getLogger(__name__)

REFUND_TICKET_CAMPAIGN = get_setting('REFUND_TICKET_CAMPAIGN', {
    Configuration.PRODUCTION: 'G6AFD6H2-G3Y1',
    Configuration.TESTING: 'WCRV7HH2-QQD1',
    Configuration.DEVELOPMENT: 'WCRV7HH2-QQD1'
}, default=None)
RESIZE_TICKET_CAMPAIGN = get_setting('RESIZE_TICKET_CAMPAIGN', {
    Configuration.PRODUCTION: 'OUPZM243-W4U',
    Configuration.TESTING: 'K5AHJ243-JO02',
    Configuration.DEVELOPMENT: 'K5AHJ243-JO02'
}, default=None)


def calculate_payment_and_refund_amount(order, blank_ids, refund_uuid):
    payment_amount = Decimal('0.00')
    full_refund_amount = Decimal('0.00')

    for passenger in order.passengers:
        if passenger.tickets[0].blank_id in blank_ids:
            payment_amount += passenger.tickets[0].payment.total

            if passenger.tickets[0].refund.amount:
                full_refund_amount += passenger.tickets[0].refund.amount

            if passenger.tickets[0].refund.refund_yandex_fee_amount:
                full_refund_amount += passenger.tickets[0].refund.refund_yandex_fee_amount

            if passenger.insurance and passenger.insurance.trust_order_id:
                payment_amount += passenger.insurance.amount
                if refund_uuid == passenger.insurance.refund_uuid:
                    full_refund_amount += passenger.insurance.amount

    return payment_amount, full_refund_amount


@translation.override('ru')
def make_refund_args(order, blank_ids, refund_uuid):
    args = make_mail_args_for_blanks(order, blank_ids)
    payment_amount, full_refund_amount = calculate_payment_and_refund_amount(order, blank_ids, refund_uuid)

    args.update({
        'payment_sum': format_money(payment_amount),
        'refund_sum': format_money(full_refund_amount),
        'fee_sum': format_money(payment_amount - full_refund_amount),
    })
    return args


def send_refund_email(refund, order=None, email=None):
    if not order:
        order = refund.order
    if not email:
        email = order.user_info.email
    log.info('Начинаем отправку письма о возврате %s', refund.uuid)
    TrainOrder.fetch_stations([order])
    args = make_refund_args(order, refund.blank_ids, refund.uuid)
    refund_payment = refund.refund_payment
    campaign = RESIZE_TICKET_CAMPAIGN if (refund_payment and refund_payment.payment_resized) else REFUND_TICKET_CAMPAIGN
    email_id = guaranteed_send_email(
        key='send_refund_{}_to_{}_at_{}'.format(refund.uuid, email, now_aware()),
        to_email=email,
        args=args,
        campaign=campaign,
        data={'order_uid': order.uid, 'refund_uuid': refund.uuid},
        log_context={'order_uid': order.uid},
        preprocessor=refund_attachment_adder
    )
    log.info('Письмо %s о возврате %s успешно поставлено в очередь гарантированной отправки', email_id, refund.uuid)


def refund_attachment_adder(email):
    refund = TrainRefund.objects.get(uuid=email.data['refund_uuid'])
    order = refund.order
    attachments = []

    if not refund.is_external:
        for ticket in order.iter_tickets():
            if ticket.blank_id not in refund.blank_ids:
                continue

            try:
                ticket_blank = download_refund_blank(order, ticket)
                blank_format = BlankFormat.PDF
                # в ticket_blank.content возможен тип bson.binary.Binary, без конверта в bytes будут падать тесты
                attachments.append(AttachmentInfo(
                    filename='{}.{}'.format(ticket.refund.operation_id, blank_format.value),
                    mime_type=blank_format.mime_type,
                    content=bytes(ticket_blank.content)
                ))
            except Exception:
                log.exception('Ошибка при обработке бланка КРС')

    if not train_order_skip_trust():
        try:
            refund_payment = refund.refund_payment
            refund_receipt = download_refund_receipt(refund_payment)
            attachments.append(AttachmentInfo(
                filename='receipt.pdf',
                mime_type='application/pdf',
                content=refund_receipt
            ))
        except Exception:
            if can_skip_trust_check(email):
                error_msg = 'Не удалось получить чек возврата за отведенное время, продолжили без него.'
                log.exception(error_msg)
                report_error_email(email, error_msg)
            else:
                raise
    email.attachments = attachments
