# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from rest_framework import status
from rest_framework.response import Response

from travel.rasp.train_api.helpers.error import render_error, ErrorType
from travel.rasp.train_api.train_purchase.core.models import TrainPurchaseSmsVerification, ErrorInfo
from travel.rasp.train_api.train_purchase.core.sms_verification import SmsThrottlingError, send_verification_sms

log = logging.getLogger(__name__)


def verify_sms(request_sms_verification, sms_verification_code, phone,
               action_name, message_template, action_data):
    if request_sms_verification:
        try:
            sms_verification = send_verification_sms(
                phone, action_name, message_template,
                action_data=action_data
            )
            log.info('Отправлен код {} по запросу на {}, action {}'.format(
                sms_verification.code,
                sms_verification.phone,
                sms_verification.action_name
            ))
        except SmsThrottlingError as e:
            log.exception('Превышен лимит сообщений для {}, action {}'.format(phone, action_name))
            return render_error(
                ErrorInfo(
                    type=ErrorType.SMS_VALIDATION_ERROR,
                    message='Нельзя запрашивать СМС слишком часто',
                    data={'canSendNextSmsAfter': e.last_sms.can_send_next_after}
                ),
                status.HTTP_429_TOO_MANY_REQUESTS,
            )

        return Response({
            'smsSent': True,
            'canSendNextSmsAfter': sms_verification.can_send_next_after,
            'smsExpiredAfter': sms_verification.expired_after
        })
    else:
        error = ErrorInfo(
            type=ErrorType.SMS_VALIDATION_ERROR,
            message='Введён неверный код',
        )

        if not sms_verification_code:
            log.info('Не указан код верификации для {}, action {}'.format(phone, action_name))
            return render_error(error, status.HTTP_412_PRECONDITION_FAILED)

        last_live_sms = TrainPurchaseSmsVerification.get_latest_live_sms(phone, action_name)

        if not last_live_sms:
            log.info('Нет актуальных сообщений для {}, action {}'.format(phone, action_name))
            return render_error(error, status.HTTP_412_PRECONDITION_FAILED)

        if last_live_sms.code != sms_verification_code:
            log.info('Указан неверный код верификации {} для {}, action {}'.format(
                sms_verification_code,
                phone, action_name,
            ))
            return render_error(error, status.HTTP_412_PRECONDITION_FAILED)

        last_live_sms.update(set__used=True)

        if last_live_sms.action_data != action_data:
            log.info('Код верификации {} верный, action_data некорректный для {}, action {}'.format(
                sms_verification_code, phone, action_name))
            return render_error(error, status.HTTP_412_PRECONDITION_FAILED)

        log.info('Успешно верифицирован код {} для {}, action {}'.format(sms_verification_code, phone, action_name))
        return None
