# coding: utf8

from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import timedelta
from decimal import Decimal

from django.conf import settings
from django.utils import translation
from django.utils.encoding import force_text
from enum import Enum
from pytz import timezone, utc
from ylog.context import log_context

from common.apps.train.models import COACH_TYPE_CHOICES
from common.data_api.billing.trust_client import train_order_skip_trust
from common.data_api.sendr.api import Campaign, Attachment
from common.dynamic_settings.default import conf
from common.email_sender import AttachmentInfo
from common.email_sender.sender import guaranteed_send_email
from common.models.geo import Settlement, Region
from common.settings.configuration import Configuration
from common.settings.utils import get_setting, define_setting
from common.utils.date import human_date, MSK_TZ
from travel.rasp.library.python.common23.date.environment import now_aware, now_utc
from common.utils.railway import get_railway_tz_by_point
from common.utils.title_generator import TitleGenerator, DefaultTitleGenerationStrategy
from common.xgettext.i18n import gettext
from travel.rasp.train_api.train_partners.base import RzhdStatus
from travel.rasp.train_api.train_partners.base.ticket_blank import download_ticket_blank, BlankFormat
from travel.rasp.train_api.train_purchase.core.models import TrainOrder, TrainRefund
from travel.rasp.train_api.train_purchase.core.utils import format_money, format_places, format_coach_number
from travel.rasp.train_api.train_purchase.utils.billing import download_order_receipt, download_refund_receipt
from travel.rasp.train_api.train_purchase.utils.email_sender_callback import report_error_email

log = logging.getLogger(__name__)


SEND_TICKET_CAMPAIGN = get_setting('SEND_TICKET_CAMPAIGN', {
    Configuration.PRODUCTION: 'SHM6YPG2-I22',
    Configuration.TESTING: 'CJGO8HH2-BHB',
    Configuration.DEVELOPMENT: 'CJGO8HH2-BHB'
}, default=None)
campaign_send = Campaign.create_rasp_campaign(SEND_TICKET_CAMPAIGN)

MAIN_PASSENGER_MIN_AGE = 18  # возраст совершеннолетия для выбора получателя письма
MOSCOW_REGION_GEO_ID = 1

define_setting('EMAIL_SENDER_TIME_AFTER_SKIP_TRUST_CHECK', default=timedelta(minutes=20))


def can_skip_trust_check(email):
    return now_utc() - email.created_at > settings.EMAIL_SENDER_TIME_AFTER_SKIP_TRUST_CHECK


def attachment_adder(email):
    order = TrainOrder.objects.get(uid=email.data['order_uid'])
    blank_format = BlankFormat.PDF
    email.attachments = [
        AttachmentInfo(filename='{}.pdf'.format(order.current_partner_data.order_num),
                       mime_type=blank_format.mime_type,
                       content=download_ticket_blank(order, order.current_partner_data.operation_id, blank_format))
    ]
    if not train_order_skip_trust():
        try:
            receipt = download_order_receipt(order)
            email.attachments.append(AttachmentInfo(
                filename='receipt.pdf',
                mime_type='application/pdf',
                content=receipt
            ))
        except Exception:
            if can_skip_trust_check(email):
                error_msg = 'Не удалось получить чек за отведенное время, продолжили без него.'
                log.exception(error_msg)
                report_error_email(email, error_msg)
            else:
                raise
    if order.insurance_auto_return_uuid:
        try:
            refund_payment = TrainRefund.objects.get(uuid=order.insurance_auto_return_uuid).refund_payment
            insurance_refund_receipt = download_refund_receipt(refund_payment)
            email.attachments.append(AttachmentInfo(
                filename='refund_receipt.pdf', mime_type='application/pdf', content=insurance_refund_receipt
            ))
        except Exception:
            if can_skip_trust_check(email):
                error_msg = 'Не удалось получить чек возврата страховок за отведенное время, продолжили без него.'
                log.exception(error_msg)
                report_error_email(email, error_msg)
            else:
                raise


def send_tickets_email(order_uid):
    order = TrainOrder.objects.get(uid=order_uid)
    with log_context(order_uid=order.uid):
        try:
            TrainOrder.fetch_stations([order])
            args = make_mail_args(order)

            email_id = guaranteed_send_email(
                key='send_tickets_{}_to_{}_at_{}'.format(order.uid, order.user_info.email, now_aware()),
                to_email=order.user_info.email,
                args=args,
                campaign=SEND_TICKET_CAMPAIGN,
                data={'order_uid': order.uid},
                log_context={'order_uid': order.uid},
                preprocessor=attachment_adder
            )
            log.info('Письмо о покупке %s поставлено в очередь гарантированной отправки', email_id)
        except Exception:
            log.exception('Не получилось поставить письмо о покупке в очередь гарантированной отправки')


def do_resend_tickets_email(order_uid, email):
    order = TrainOrder.objects.get(uid=order_uid)
    with log_context(order_uid=order.uid):
        _send_tickets_email(order, campaign_send, email)


def full_passenger_name(passenger):
    return '{} {}'.format(passenger.first_name, passenger.last_name)


def get_main_passenger(order):
    try:
        return next(passenger for passenger in order.passengers if passenger.age >= MAIN_PASSENGER_MIN_AGE)
    except StopIteration:
        return order.passengers[0]


def get_train_title(order):
    generation_strategy = DefaultTitleGenerationStrategy()
    title_dict = generation_strategy.build_title_dict(None, (order.station_from, order.station_to))
    return TitleGenerator.L_title(
        title_dict, prefetched_points=generation_strategy.prefetched_points, lang='ru'
    )


class OrderRegistrationStatus(Enum):
    DISABLED = 0
    ENABLED = 1
    MIXED = 2

    @classmethod
    def from_order(cls, order):
        enabled_statuses = {not ticket.pending and ticket.rzhd_status == RzhdStatus.REMOTE_CHECK_IN
                            for ticket in order.iter_tickets()}

        if len(enabled_statuses) > 1:
            return cls.MIXED

        return cls.ENABLED if enabled_statuses == {True} else cls.DISABLED


def get_tickets_price(order):
    return sum((ticket.payment.total for ticket in order.iter_tickets()), Decimal(0))


def get_insurances_price(order):
    return sum((p.insurance.amount for p in order.passengers if p.insurance and p.insurance.trust_order_id), Decimal(0))


def make_arrival_departure_args(order, departure_tz, arrival_tz, suffix='', make_tzname=False):
    departure_dt = utc.localize(order.departure).astimezone(departure_tz)
    arrival_dt = utc.localize(order.arrival).astimezone(arrival_tz)
    args = {
        'departure_date{}'.format(suffix): human_date(departure_dt),
        'departure_time{}'.format(suffix): force_text(departure_dt.strftime('%H:%M')),
        'departure_date_dot{}'.format(suffix): force_text(departure_dt.strftime('%d.%m.%Y')),
        'arrival_date{}'.format(suffix): human_date(arrival_dt),
        'arrival_time{}'.format(suffix): force_text(arrival_dt.strftime('%H:%M')),
    }
    if make_tzname:
        args.update({'departure_tzname{}'.format(suffix): gettext('by_tz_{}'.format(departure_tz.zone)),
                     'arrival_tzname{}'.format(suffix): gettext('by_tz_{}'.format(arrival_tz.zone))})
    return args


def is_travel_terminal(order):
    return order.source and order.source.terminal == 'travel'


def get_front_url(is_travel):
    if is_travel:
        return conf.TRAVEL_FRONT_URL
    return conf.TRAIN_FRONT_URL


@translation.override('ru')
def make_mail_common_args(order):
    is_travel = is_travel_terminal(order)
    args = {
        'coach_number': format_coach_number(order.coach_number),
        'coach_type': force_text(next(title for code, title in COACH_TYPE_CHOICES if code == order.coach_type.value)),
        'main_name': full_passenger_name(get_main_passenger(order)),
        'main_first_name': get_main_passenger(order).first_name,
        'order_uid': order.uid,
        'registration_status': OrderRegistrationStatus.from_order(order).name,
        'station_from_title': order.station_from.title,
        'station_to_title': order.station_to.title,
        'ticket_number': order.current_partner_data.order_num,
        'train_number': order.train_ticket_number,
        'train_title': get_train_title(order),
        'front_url': get_front_url(is_travel),
        'is_moscow_region': _is_moscow_region(order),
        'is_travel': is_travel,
    }

    args.update(make_arrival_departure_args(order, get_railway_tz_by_point(order.station_from),
                                            get_railway_tz_by_point(order.station_to), '_rw', True))
    args.update(make_arrival_departure_args(order, timezone(order.station_from.time_zone),
                                            timezone(order.station_to.time_zone), '_local'))
    args.update(make_arrival_departure_args(order, MSK_TZ, MSK_TZ))

    return args


@translation.override('ru')
def make_mail_args(order):
    args = make_mail_common_args(order)
    tickets_price = get_tickets_price(order)
    insurances_price = get_insurances_price(order)
    args.update({
        'passengers': [{'name': full_passenger_name(p),
                        'first_name': p.first_name,
                        'place': format_places([pl for t in p.tickets for pl in t.places])}
                       for p in order.passengers],
        'tickets': [{'blank_id': t.blank_id} for t in order.iter_tickets()],
        'initial_order_price': format_money(tickets_price + insurances_price),
        'order_price': format_money(tickets_price + (0 if order.insurance_auto_return_uuid else insurances_price)),
        'insurance_price': format_money(insurances_price),
        'insurance_auto_return': bool(order.insurance_auto_return_uuid),
    })
    return args


@translation.override('ru')
def make_mail_args_for_blanks(order, blank_ids):
    args = make_mail_common_args(order)
    passengers_in_blanks = [p for p in order.passengers if any(t.blank_id in blank_ids for t in p.tickets)]
    args.update({
        'passengers': [{
            'name': full_passenger_name(p),
            'first_name': p.first_name,
            'place': format_places([pl for t in p.tickets if t.blank_id in blank_ids for pl in t.places])
        } for p in passengers_in_blanks],
        'tickets': [{'blank_id': bid} for bid in blank_ids],
    })
    return args


def _send_tickets_email(order, campaign, email):
    blank_format = BlankFormat.PDF
    if train_order_skip_trust():
        receipt = b'pdfdata'
    else:
        receipt = download_order_receipt(order)
    attachments = [
        Attachment(filename='{}.pdf'.format(order.current_partner_data.order_num),
                   mime_type=blank_format.mime_type,
                   content=download_ticket_blank(order, order.current_partner_data.operation_id, blank_format)),
        Attachment(filename='receipt.pdf',
                   mime_type='application/pdf',
                   content=receipt),
    ]
    if order.insurance_auto_return_uuid:
        refund_payment = TrainRefund.objects.get(uuid=order.insurance_auto_return_uuid).refund_payment
        insurance_refund_receipt = download_refund_receipt(refund_payment)
        attachments.append(Attachment(
            filename='refund_receipt.pdf', mime_type='application/pdf', content=insurance_refund_receipt
        ))
    TrainOrder.fetch_stations([order])
    args = make_mail_args(order)
    campaign.send(email, args, attachments=attachments)


def _is_moscow_region(order):
    user_geo_id = order.user_info.region_id
    is_moscow_region = user_geo_id == MOSCOW_REGION_GEO_ID
    if not is_moscow_region:
        values = Settlement.objects.filter(_geo_id=user_geo_id).values('region_id').first()
        is_moscow_region = values['region_id'] == Region.MOSCOW_REGION_ID if values else False
    return is_moscow_region
