# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

import celery
from django.utils import translation
from pytz import timezone
from ylog.context import log_context

from common.data_api.url_shortener.client import ClckClient
from common.data_api.yasms.client import YaSMSClient
from travel.rasp.train_api.train_purchase.core.models import TrainOrder
from travel.rasp.train_api.train_purchase.utils.tickets_email import make_arrival_departure_args, is_travel_terminal, get_front_url

log = logging.getLogger(__name__)


def make_url(order):
    is_travel = is_travel_terminal(order)
    if is_travel:
        return 'https://{front_url}/my/order/{order_uid}/'.format(
            front_url=get_front_url(is_travel), order_uid=order.uid)
    return 'https://{front_url}/orders/{order_uid}/'.format(front_url=get_front_url(is_travel), order_uid=order.uid)


@translation.override('ru')
def make_order_sms_args(order):
    args = make_arrival_departure_args(order, timezone(order.station_from.time_zone),
                                       timezone(order.station_to.time_zone))
    args.update({
        'station_from_title': order.station_from.get_short_title(),
        'station_to_title': order.station_to.get_short_title(),
        'train_number': order.train_ticket_number,
    })
    return args


@translation.override('ru')
def make_order_purchase_sms_args(order):
    args = make_order_sms_args(order)
    url = make_url(order)
    args.update({
        'short_url': ClckClient(url_type=ClckClient.LONG_TYPE).shorten(url),
    })
    return args


@celery.shared_task
def send_tickets_sms(order_uid):
    order = TrainOrder.objects.get(uid=order_uid)
    with log_context(order_uid=order.uid):
        try:
            TrainOrder.fetch_stations([order])

            message = (
                'Ваш билет на поезд {train_number} {departure_date}: {short_url}'
                if len(list(order.iter_tickets())) == 1 else
                'Ваши билеты на поезд {train_number} {departure_date}: {short_url}'
            ).format(**make_order_purchase_sms_args(order))

            log.info('Посылаем SMS  о покупке: %s "%s"', order.user_info.phone, message)
            yasms_id = YaSMSClient().sendsms(message, phone=order.user_info.phone)
            log.info('SMS %s отправлено', yasms_id)
            return yasms_id
        except Exception:
            log.exception('Не получилось отправить SMS')
