# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from functools import wraps

from django.conf import settings
from rest_framework import status
from rest_framework.response import Response

from common.settings.utils import define_setting
from common.data_api.tvm.instance import tvm_factory

log = logging.getLogger(__name__)

define_setting('CHECK_TVM_SERVICE_TICKET', default=True)


class TvmServiceId(object):
    RASP_DEVELOPERS = 2001121

    HOTELS_TESTING = 2002548
    HOTELS_PRODUCTION = 2002546

    TRAVEL_FRONT_TESTING = 2002416
    TRAVEL_FRONT_PRODUCTION = 2002364


def check_service_ticket(allowed_tvm_service_ids):
    def decorator(view_func):
        @wraps(view_func)
        def wrapper(request, *args, **kwargs):
            if settings.CHECK_TVM_SERVICE_TICKET:
                is_allowed_service, error = _check_service_ticket(request, allowed_tvm_service_ids)
                if not is_allowed_service:
                    return Response({
                        'errors': {'TVM error': error}
                    }, status=status.HTTP_403_FORBIDDEN)
            return view_func(request, *args, **kwargs)
        return wrapper
    return decorator


def _check_service_ticket(request, allowed_tvm_service_ids):
    try:
        ticket = request.META.get('HTTP_X_YA_SERVICE_TICKET')
        if not ticket:
            return False, 'TVM service ticket is not provided.'
        service_ticket = tvm_factory.get_provider().check_service_ticket(ticket=ticket)
        if not service_ticket:
            return False, 'Service ticket is undefined.'
        is_allowed_service = service_ticket.src in allowed_tvm_service_ids
        return is_allowed_service, '' if is_allowed_service else 'Service ticket is not allowed.'
    except Exception:
        error_message = 'Unknown TVM error.'
        log.exception(error_message)
        return False, error_message
