# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import re

from django.conf import settings
from django.utils.encoding import force_text
from django.utils.translation import get_language
from mongoengine import Q
from pymongo.collation import CollationStrength
from rest_framework import status
from rest_framework.decorators import list_route, api_view
from rest_framework.response import Response
from rest_framework.routers import SimpleRouter
from rest_framework.viewsets import ViewSet
from ylog.context import log_context

from common.dev_tools.swagger import swagger_aware_view_set
from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from travel.rasp.library.python.common23.date import environment
from common.workflow import registry
from common.workflow.registry import run_process
from common.data_api.tvm.instance import tvm_factory
from travel.rasp.train_api.train_partners.base import RzhdStatus
from travel.rasp.train_api.train_purchase.core.enums import OrderStatus, TrainPartner, TravelOrderStatus, InsuranceStatus
from travel.rasp.train_api.train_purchase.core.models import TrainOrder, RefundStatus, TrainRefund, Payment
from travel.rasp.train_api.train_purchase.serialization import (
    OrderSchema, FindOrderSchema, RefundRequestSchema, ActivePartnersResponseSchema, TakeoutOrderDetailsSchema,
    UserOrdersRequestSchema, RunBookingSchema, RetrieveRequestSchema, OrderBulkSchema
)
from travel.rasp.train_api.train_purchase.utils.decorators import order_detail_route
from travel.rasp.train_api.train_purchase.utils.order import update_tickets_statuses, send_event_to_order, send_event_to_payment
from travel.rasp.train_api.train_purchase.utils.pagination import MultipleQuerySetsPagination
from travel.rasp.train_api.train_purchase.utils.sms_verification import verify_sms
from travel.rasp.train_api.train_purchase.utils.tickets_sms import make_order_sms_args
from travel.rasp.train_api.train_purchase.utils.tvm import check_service_ticket, TvmServiceId
from travel.rasp.train_api.train_purchase.workflow.booking import TRAIN_BOOKING_PROCESS
from travel.rasp.train_api.train_purchase.workflow.ticket_refund import TICKET_REFUND_PROCESS
from travel.rasp.train_api.train_purchase.workflow.user_events import PaymentUserEvents, TrainBookingUserEvents

REFUND_SMS_ACTION_NAME = 'REFUND_SMS'
USER_ORDERS_PAGE_SIZE = 100
log = logging.getLogger(__name__)

define_setting('TAKEOUT_ALLOWED_SRC', {
    Configuration.PRODUCTION: {
        2001121,  # rasp-developers
        2009785,  # takeout-production
    },
    Configuration.TESTING: {
        2001121,  # rasp-developers
        2009783,  # takeout-testing
    }
}, default={
    2001121,  # rasp-developers
})


@swagger_aware_view_set
class OrderViewSet(ViewSet):
    @order_detail_route(log_call=False, builtin=True)
    def retrieve(self, request, order):
        """Получить заказ
        ---
        parameters:
          - in: query
            name: query_params
            schema:
                $ref: 'RetrieveRequestSchema'
          - in: path
            name: pk
            required: true
            description: Uid заказа
            schema:
                type: string
        responses:
            200:
                description: Заказ
                schema:
                    properties:
                        order:
                            $ref: 'OrderSchema'
        """
        query, errors = RetrieveRequestSchema().load(request.GET)
        if errors:
            return Response({
                'errors': errors,
            }, status=status.HTTP_400_BAD_REQUEST)

        tickets_pending = any(t.pending for t in order.iter_tickets())
        if order.status == OrderStatus.DONE or tickets_pending:
            update_tickets_statuses(
                order, update_order=tickets_pending, set_personal_data=True, set_order_warnings=True,
                first_actual_warning_only=query['first_actual_warning_only'],
            )
        return Response({
            'order': OrderSchema().dump(order).data
        })

    @order_detail_route(methods=['post'])
    def cancel(self, request, order):
        """
        Отмена заказа
        ---
        parameters:
          - in: path
            name: pk
            required: true
            description: Uid заказа
            schema:
              type: string
        responses:
            200:
                description: все получилось
        """
        send_event_to_payment(order.current_billing_payment, PaymentUserEvents.CANCEL, allow_send_to_empty_process=True)
        return Response({
            'ok': True
        })

    @order_detail_route(methods=['post'])
    def retry_payment(self, request, order):
        """
        Повторить попытку оплаты
        ---
        parameters:
          - in: path
            name: pk
            required: true
            description: Uid заказа
            schema:
              type: string
        responses:
            200:
                description: все получилось
            406:
                description: заказ отменен
            409:
                description: статус заказа не подходит для выполения операции
        """
        if order.current_partner_data.is_order_cancelled:
            return Response({
                'errors': {'order': 'Order is cancelled, can not do payment'},
            }, status=status.HTTP_406_NOT_ACCEPTABLE)

        filter_params = {
            'uid': order.uid,
            'status': OrderStatus.PAYMENT_FAILED,
        }
        update_spec = {
            'set__status': OrderStatus.RESERVED,
            'set__travel_status': TravelOrderStatus.RESERVED,
        }
        Payment.objects.create(order_uid=order.uid)
        update_result = TrainOrder.objects.filter(**filter_params).update_one(**update_spec)
        if not update_result:
            return Response({
                'errors': {'order': 'Bad status, can not do payment'},
            }, status=status.HTTP_409_CONFLICT)

        order.reload()
        send_event_to_order(order, TrainBookingUserEvents.RETRY_PAYMENT)

        return Response({
            'ok': True
        })

    @order_detail_route(methods=['post'])
    def refund(self, request, order):
        """
        Начать возврат
        ---
        parameters:
          - in: body
            name: post_schema
            schema:
                $ref: 'RefundRequestSchema'
          - in: path
            name: pk
            required: true
            description: Uid заказа
            schema:
                type: string
        responses:
            200:
                description: Заказ
                schema:
                    properties:
                        order:
                            $ref: 'OrderSchema'
            400:
                schema:
                    properties:
                        errors:
                            type: object
            409:
                description: статус заказа не подходит для выполения операции
        """
        if not order.partner.is_active:
            return Response({
                'errors': 'Partner is not active',
            }, status=status.HTTP_400_BAD_REQUEST)

        refund_request, errors = RefundRequestSchema().load(request.data)
        if errors:
            return Response({
                'errors': errors,
            }, status=status.HTTP_400_BAD_REQUEST)

        if not is_refund_possible(order, refund_request['blank_ids']):
            return Response({
                'errors': {'blankIds': 'Some of blank_ids are in invalid status'},
            }, status=status.HTTP_400_BAD_REQUEST)

        TrainOrder.fetch_stations([order])
        refund_tickets = [t for t in order.iter_tickets() if t.blank_id in refund_request['blank_ids']]
        if len(refund_tickets) > 1:
            message_template = '{{code}} — код для возврата билетов на поезд {train_number} {departure_date}'
        else:
            message_template = '{{code}} — код для возврата билета на поезд {train_number} {departure_date}'
        message_template = message_template.format(**make_order_sms_args(order))

        response = verify_sms(
            request_sms_verification=refund_request.get('request_sms_verification'),
            sms_verification_code=refund_request.get('sms_verification_code'),
            phone=order.user_info.phone, message_template=message_template,
            action_name=generate_refund_sms_action_name(refund_request['blank_ids']),
            action_data={
                'uid': order.uid,
                'blank_ids': sorted(refund_request['blank_ids'])
            }
        )
        if response is not None:
            return response

        if order.status != OrderStatus.DONE:
            return Response({
                'errors': {'order': 'Bad status, can not do refund'},
            }, status=status.HTTP_409_CONFLICT)

        try:
            refund = TrainRefund.objects.create(
                order_uid=order.uid,
                is_active=True,
                status=RefundStatus.NEW,
                blank_ids=refund_request['blank_ids'],
                user_info=refund_request['user_info'],
                created_at=environment.now_utc(),
            )
        except Exception:
            log.exception('Ошибка при создании возврата')
            return Response({
                'errors': {'order': "Can't create refund"},
            }, status=status.HTTP_409_CONFLICT)

        registry.run_process.apply_async([TICKET_REFUND_PROCESS, str(refund.id), {'order_uid': order.uid}])

        return Response({
            'order': OrderSchema().dump(order).data
        })

    @list_route(methods=['get'])
    def find(self, request):
        """
        Найти заказ
        ---
        parameters:
          - in: query
            name: query_params
            schema:
                $ref: 'FindOrderSchema'
        responses:
            200:
                description: Найденный заказ
                schema:
                    properties:
                        order:
                            $ref: 'OrderSchema'
        """
        order_query, errors = FindOrderSchema().load(request.GET)

        if errors:
            return Response({
                'errors': errors,
            }, status=status.HTTP_400_BAD_REQUEST)

        # ищем без учёта регистра и диакритических знаков
        if order_query['email']:
            orders = list(TrainOrder.objects.filter(
                user_info__email=order_query['email'],
                partner_data_history__order_num=order_query['order_num'],
            ).collation(locale=get_language(), strength=CollationStrength.PRIMARY))
        elif order_query['phone']:
            orders = list(TrainOrder.objects.filter(
                user_info__reversed_phone=order_query['phone'][::-1],
                partner_data_history__order_num=order_query['order_num'],
            ))
        else:
            return Response({
                'errors': 'Require one of email or phone',
            }, status=status.HTTP_400_BAD_REQUEST)

        # проверка вместо get
        if len(orders) > 1:
            raise TrainOrder.MultipleObjectsReturned
        if len(orders) == 0:
            return Response({
                'errors': {'order': 'Order was not found'},
            }, status=status.HTTP_404_NOT_FOUND)

        order = orders[0]

        return Response({
            'order': OrderSchema().dump(order).data
        })

    @order_detail_route()
    def run_booking(self, request, order):
        """
        Запуск процесса заказа страховки, подтверждения бронирования и переход к оплате
        ---
        parameters:
          - in: query
            name: query_params
            schema:
                $ref: 'RunBookingSchema'
          - in: path
            name: pk
            required: true
            description: Uid заказа
            schema:
                type: string
        responses:
            200:
                description: ok
            500:
                description: ошибка исполнения
        """
        with log_context(order_uid=order.uid):
            try:
                run_request, errors = RunBookingSchema().load(request.query_params)
                if errors:
                    return Response(errors, status=status.HTTP_400_BAD_REQUEST)
                with_insurance = run_request['with_insurance']

                order.update(**{
                    'set__insurance__status': InsuranceStatus.ACCEPTED if with_insurance else InsuranceStatus.DECLINED,
                    'set__process__suspended': False,
                })
                try:
                    run_process.apply_async([TRAIN_BOOKING_PROCESS, str(order.id), {'order_uid': order.uid}])
                except Exception:
                    log.warning('Error in run_process.apply_async', exc_info=True)

            except Exception:
                log.exception('Error in run_booking')
                return Response({'errors': 'Error in run_booking'},
                                status=status.HTTP_500_INTERNAL_SERVER_ERROR)

            return Response({'ok': True})


def is_refund_possible(order, refund_blank_ids):
    blank_id_to_rzhd_status = {
        t.blank_id: RzhdStatus(t.rzhd_status) for t in order.iter_tickets()
        if t.rzhd_status is not None
    }
    return all(
        blank_id in blank_id_to_rzhd_status and blank_id_to_rzhd_status[blank_id].is_refundable()
        for blank_id in refund_blank_ids
    )


orders_router = SimpleRouter()
orders_router.register('train-purchase/orders', OrderViewSet, base_name='orders')


define_setting('USER_ORDERS_ALLOWED_SERVICE_IDS', {
    Configuration.PRODUCTION: [
        TvmServiceId.RASP_DEVELOPERS,
        TvmServiceId.TRAVEL_FRONT_PRODUCTION,
        TvmServiceId.HOTELS_PRODUCTION,
    ],
    Configuration.TESTING: [
        TvmServiceId.RASP_DEVELOPERS,
        TvmServiceId.TRAVEL_FRONT_TESTING,
        TvmServiceId.HOTELS_TESTING,
    ],
}, default=[TvmServiceId.RASP_DEVELOPERS])


@api_view(['GET'])
@check_service_ticket(settings.USER_ORDERS_ALLOWED_SERVICE_IDS)
def user_orders(request):
    """
    Ручка по заказам пользователя для ЛК Путешествий, ожидает user_ticket в заголовке
    ---
    parameters:
      - in: query
        name: query_params
        schema:
            $ref: 'UserOrdersRequestSchema'
    responses:
        200:
            description: информация из траста о платеже
            schema:
               properties:
                  results:
                    type: array
                    items:
                      $ref: 'OrderSchema'
                  count:
                    description: количество заказов по запросу, без учета разбиения на страницы
                    type: integer
                  counts:
                    description: количество заказов пользователя по статусам
                    type: object
    """
    user_uid = _get_user_uid(request)
    if not user_uid:
        return Response({
            'errors': {'TVM error': 'Error in user ticket.'}
        }, status=status.HTTP_403_FORBIDDEN)

    order_query, errors = UserOrdersRequestSchema().load(request.GET)
    if errors:
        return Response({
            'errors': errors,
        }, status=status.HTTP_400_BAD_REQUEST)

    today = environment.now_utc().date()
    paginator = MultipleQuerySetsPagination()
    paginator.default_limit = USER_ORDERS_PAGE_SIZE
    filters = Q(user_info__uid=user_uid) & Q(travel_status__in=order_query['travel_statuses']) & Q(removed__ne=True)
    find_string = order_query['find_string']
    if find_string:
        like_query = {"$regex": re.escape(find_string), '$options': '-i'}
        filters &= (
            Q(passengers__last_name=like_query)
            | Q(passengers__first_name=like_query)
            | Q(passengers__patronymic=like_query)
            | Q(partner_data_history__order_num=like_query)
            | Q(route_info__start_station__settlement_title=like_query)
            | Q(route_info__end_station__settlement_title=like_query)
            | Q(route_info__from_station__title=like_query)
            | Q(route_info__to_station__title=like_query)
        )
    actual_orders = TrainOrder.objects.filter(filters & Q(departure__gte=today)).order_by('departure', 'uid')
    past_orders = TrainOrder.objects.filter(filters & Q(departure__lt=today)).order_by('-departure', 'uid')
    response = paginator.get_paginated_response(
        OrderBulkSchema().dump(paginator.paginate_querysets((actual_orders, past_orders), request), many=True).data
    )
    if order_query['get_count_by_statuses']:
        counts = {row['_id']: row['count'] for row in TrainOrder.objects.aggregate(*[
            {'$match': {'user_info.uid': user_uid}},
            {'$group': {'_id': '$travel_status', 'count': {'$sum': 1}}},
            {'$project': {'_id': 1, 'count': 1}}
        ])}
        response.data['counts'] = counts
    return response


def _get_user_uid(request):
    user_uid = None
    try:
        user_ticket = tvm_factory.get_provider().check_user_ticket(request.META.get('HTTP_X_YA_USER_TICKET'))
        user_uid = str(user_ticket.default_uid)
    except Exception as error:
        log.warning('TVM error. %s', force_text(error))
        if settings.YANDEX_ENVIRONMENT_TYPE in [Configuration.TESTING, Configuration.DEVELOPMENT]:
            user_uid = request.query_params.get('debug_user_uid')
    return user_uid


@api_view(['GET'])
def get_active_partners(request):
    """
    Ручка отдает партнеров, которые включены в настройках и у которых есть действующий контракт
    ---
    responses:
        200:
            schema:
                $ref: 'ActivePartnersResponseSchema'
    """
    return Response(
        ActivePartnersResponseSchema().dump({
            'partner_codes': [partner.value for partner in TrainPartner if partner.is_active]
        }).data)


@api_view(['POST'])
def takeout_orders(request):
    """
    Ручка отдает заказы пользователя по запросу Takeout
    ---
    parameters:
      - in: body
        name: post_schema
        schema:
            properties:
                uid:
                    description: uid пользователя
                    type: string
    responses:
        200:
            schema:
               properties:
                  status:
                    description: error/no_data/ok
                    type: string
                  error:
                    description: текст ошибки при ошибке
                    type: string
                  data:
                    description: данные по заказам в случае успеха
                    type: object
                    properties:
                      orders.json:
                        type: array
                        items:
                          $ref: 'TakeoutOrderDetailsSchema'
    """
    class PreconditionsError(Exception):
        pass

    try:
        try:
            ticket = request.META.get('HTTP_X_YA_SERVICE_TICKET')
            if not ticket:
                raise Exception('Service ticket is undefined.')
            service_ticket = tvm_factory.get_provider().check_service_ticket(ticket=ticket)
            if service_ticket.src not in settings.TAKEOUT_ALLOWED_SRC:
                raise Exception('Service ticket is not allowed.')
        except Exception as tvm_error:
            raise PreconditionsError('TVM error: {}'.format(force_text(tvm_error)))

        uid = request.POST.get('uid')
        if not uid:
            raise PreconditionsError('UID is undefined.')

        queryset = TrainOrder.objects.filter(user_info__uid=uid).order_by('id')

        if not queryset.count():
            # код нужен 200 чтобы не ретраили https://st.yandex-team.ru/TRAINS-4184#5d825ae54b92d7001c033d93
            return Response({'status': 'no_data'}, status=status.HTTP_200_OK)

        return Response({
            'status': 'ok',
            'data': {
                'orders.json': TakeoutOrderDetailsSchema().dumps(queryset, many=True).data,
            },
        })
    except PreconditionsError as e:
        log.exception('Ошибка параметров запроса на выгрузку Takeout')
        # код нужен 200 чтобы не ретраили https://st.yandex-team.ru/TRAINS-4184#5d825ae54b92d7001c033d93
        return Response({'status': 'error', 'error': force_text(e)}, status=status.HTTP_200_OK)
    except Exception:
        error_message = 'Неизвестная ошибка выгрузки Takeout'
        log.exception(error_message)
        # если ошибка наша, то код 500, пусть ретраят, в идеале на наших мониторингах должно зажечься
        return Response({'status': 'error', 'error': error_message}, status=status.HTTP_500_INTERNAL_SERVER_ERROR)


def generate_refund_sms_action_name(blank_ids):
    if blank_ids:
        return '{}-{}'.format(REFUND_SMS_ACTION_NAME, '-'.join(sorted(blank_ids)))
    else:
        return REFUND_SMS_ACTION_NAME
