# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from datetime import datetime
from decimal import Decimal

from pytz import UTC

from common.apps.train.models import TariffInfo
from common.apps.train_order.enums import CoachType
from common.utils.namedtuple import namedtuple_with_defaults
from common.workflow.utils import get_by_dotted_path
from travel.rasp.train_api.train_purchase.core.enums import (
    OrderStatus, TrainPartner, GenderChoice, TrainPurchaseSource, TravelOrderStatus
)
from travel.rasp.train_api.train_purchase.core.models import RefundStatus

DATETIME_FORMAT = '%Y-%m-%dT%H:%M:%SZ'
START_EPOCH_NAIVE = datetime(1970, 1, 1)
START_EPOCH = UTC.localize(START_EPOCH_NAIVE)


class Extractor(namedtuple_with_defaults('Extractor', ('name', 'path_or_getter', 'to_value_function'),
                                         defaults={'to_value_function': None})):
    def get_value(self, obj):
        value = None
        if callable(self.path_or_getter):
            value = self.path_or_getter(obj)
        else:
            try:
                value = get_by_dotted_path(obj, self.path_or_getter, default=None)
            except Exception:
                pass
        if value is not None and self.to_value_function:
            value = self.to_value_function(value)

        return value


def _dt_to_str(dt):
    return dt.strftime(DATETIME_FORMAT) if dt else None


def _dt_to_timestamp(dt):
    return int((dt - START_EPOCH_NAIVE).total_seconds()) if dt else None


def _aware_dt_to_timestamp(dt):
    return int((dt - START_EPOCH).total_seconds()) if dt else None


def _create_extractors():
    return (
        [
            Extractor('uid', 'uid'),
            Extractor('created_at', 'id.generation_time', _aware_dt_to_timestamp),
            Extractor('order_number', 'current_partner_data.order_num'),
            Extractor('status', 'status', lambda v: OrderStatus(v).value),
            Extractor('partner', 'partner', lambda v: TrainPartner(v).value),
            Extractor('train_number', 'train_number'),
            Extractor('train_ticket_number', 'train_ticket_number'),
            Extractor('train_name', 'train_name'),
            Extractor('departure', 'departure', _dt_to_timestamp),
            Extractor('departure_str', 'departure', _dt_to_str),
            Extractor('arrival', 'arrival', _dt_to_timestamp),
            Extractor('arrival_str', 'arrival', _dt_to_str),
            Extractor('coach_type', 'coach_type', lambda v: CoachType(v).value),
            Extractor('gender', 'gender', lambda v: GenderChoice(v).value),
            Extractor('two_storey', 'two_storey'),
            Extractor('coach_owner', 'coach_owner'),
            Extractor('displayed_coach_owner', 'displayed_coach_owner'),
            Extractor('scheme_id', lambda o: str(o.scheme_id) if o.scheme_id else None),
            Extractor('service_class', 'rebooking_info.service_class'),
            Extractor('international_service_class', 'rebooking_info.international_service_class'),
            Extractor('rebooking_available', lambda o: bool(o.rebooking_info and o.rebooking_info.enabled)),
            Extractor('coach_number', 'coach_number'),
            Extractor('travel_status', 'travel_status', lambda v: TravelOrderStatus(v).value),
            Extractor('finished_at', 'finished_at', _dt_to_timestamp),
            Extractor('finished_at_str', 'finished_at', _dt_to_str),
            Extractor('insurance_auto_return', lambda o: bool(o.insurance_auto_return_uuid)),
            # user_info
            Extractor('user_ip', 'user_info.ip'),
            Extractor('user_passport_uid', 'user_info.uid'),
            Extractor('user_region_id', 'user_info.region_id'),
            Extractor('user_is_mobile', 'user_info.is_mobile'),
            Extractor('user_yandex_uid', 'user_info.yandex_uid'),
            # order_amount
            Extractor('order_amount', lambda o: sum(p.total for p in o.passengers)),
            Extractor('profit_amount', lambda o: sum(p.total_fee for p in o.passengers)),
            Extractor('total_tariff_amount', lambda o: sum(p.total_ticket_amount for p in o.passengers)),
            Extractor('total_service_amount', lambda o: sum(p.total_service_amount for p in o.passengers)),
            Extractor('total_fee_amount', lambda o: sum(p.total_tickets_fee for p in o.passengers)),
            Extractor('total_insurance_amount', lambda o: sum(p.total_insurance for p in o.passengers)),
            Extractor('total_insurance_profit_amount', lambda o: sum(p.insurance_fee for p in o.passengers)),
            Extractor('total_partner_fee_amount', lambda o: sum(p.total_partner_fee for p in o.passengers)),
            Extractor('total_refund_ticket_amount',
                      lambda o: sum(t.refund.amount or Decimal(0) for t in o.refunded_tickets)),
            Extractor('total_refund_fee_amount',
                      lambda o: sum(t.refund.refund_yandex_fee_amount or Decimal(0) for t in o.refunded_tickets)),
            Extractor('total_refund_insurance_amount', lambda o: sum(i.amount for i in o.refunded_insurances)),
            Extractor('total_partner_refund_fee_amount',
                      lambda o: sum(t.payment.partner_refund_fee for t in o.refunded_tickets)),
            # counts
            Extractor('passengers_count', lambda o: len(o.passengers)),
            Extractor('adult_passengers_count',
                      lambda o: sum(1 for t in o.iter_tickets() if t.tariff_info_code == TariffInfo.FULL_CODE)),
            Extractor('children_with_seats_count',
                      lambda o: sum(1 for t in o.iter_tickets() if t.tariff_info_code == TariffInfo.CHILD_CODE)),
            Extractor('children_without_seats_count',
                      lambda o: sum(1 for t in o.iter_tickets() if t.tariff_info_code == TariffInfo.BABY_CODE)),
            Extractor('tickets_with_places_count', lambda o: sum(1 for t in o.iter_tickets() if t.places)),
            Extractor('tickets_without_places_count', lambda o: sum(1 for t in o.iter_tickets() if not t.places)),
            Extractor('requested_ticket_count', lambda o: len(list(o.iter_tickets()))),
            Extractor('total_ticket_count',
                      lambda o: len(list(o.iter_tickets())) if o.status == OrderStatus.DONE else 0),
            Extractor('refunded_ticket_count', lambda o: len(o.refunded_tickets)),
            Extractor('refunds_count', lambda o: len(o.done_refunds)),
            Extractor('bought_insurance_count', lambda o: len(o.iter_insurances())),
            # payment
            Extractor('payment_uid', 'payment.uid'),
            Extractor('payment_purchase_token', 'payment.purchase_token'),
            Extractor('payment_trust_created_at', 'payment.trust_created_at', _dt_to_timestamp),
            Extractor('payment_trust_created_at_str', 'payment.trust_created_at', _dt_to_str),
            Extractor('payment_status', 'payment.status'),
            Extractor('payment_use_deferred_clearing', 'payment.use_deferred_clearing'),
            Extractor('payment_hold_at', 'payment.hold_at', _dt_to_timestamp),
            Extractor('payment_hold_at_str', 'payment.hold_at', _dt_to_str),
            Extractor('payment_clear_at', 'payment.clear_at', _dt_to_timestamp),
            Extractor('payment_clear_at_str', 'payment.clear_at', _dt_to_str),
            Extractor('purchase_tokens_history', 'purchase_tokens_history',
                      lambda purchase_tokens: ', '.join((t if t is not None else 'None' for t in purchase_tokens))),
            Extractor('payment_attempts', lambda o: len(o.payments)),
            # route_info
            Extractor('route_info_start_station_id', 'route_info.start_station.id'),
            Extractor('route_info_start_station_title', 'route_info.start_station.title'),
            Extractor('route_info_start_station_settlement_title', 'route_info.start_station.settlement_title'),
            Extractor('route_info_start_station_departure', 'route_info.start_station.departure', _dt_to_timestamp),
            Extractor('route_info_start_station_departure_str', 'route_info.start_station.departure', _dt_to_str),
            Extractor('route_info_end_station_id', 'route_info.end_station.id'),
            Extractor('route_info_end_station_title', 'route_info.end_station.title'),
            Extractor('route_info_end_station_settlement_title', 'route_info.end_station.settlement_title'),
            Extractor('route_info_end_station_departure', 'route_info.end_station.departure', _dt_to_timestamp),
            Extractor('route_info_end_station_departure_str', 'route_info.end_station.departure', _dt_to_str),
            Extractor('route_info_from_station_id', 'route_info.from_station.id'),
            Extractor('route_info_from_station_title', 'route_info.from_station.title'),
            Extractor('route_info_from_station_settlement_title', 'route_info.from_station.settlement_title'),
            Extractor('route_info_from_station_departure', 'route_info.from_station.departure', _dt_to_timestamp),
            Extractor('route_info_from_station_departure_str', 'route_info.from_station.departure', _dt_to_str),
            Extractor('route_info_to_station_id', 'route_info.to_station.id'),
            Extractor('route_info_to_station_title', 'route_info.to_station.title'),
            Extractor('route_info_to_station_settlement_title', 'route_info.to_station.settlement_title'),
            Extractor('route_info_to_station_departure', 'route_info.to_station.departure', _dt_to_timestamp),
            Extractor('route_info_to_station_departure_str', 'route_info.to_station.departure', _dt_to_str),
            # orders_created
            Extractor('orders_created', 'orders_created', lambda orders_created: ', '.join(orders_created)),
            # partner data
            Extractor('partner_data_im_order_id', 'current_partner_data.im_order_id'),
            Extractor('partner_data_operation_id', 'current_partner_data.operation_id'),
            Extractor('partner_data_expire_set_er', 'current_partner_data.expire_set_er', _dt_to_timestamp),
            Extractor('partner_data_expire_set_er_str', 'current_partner_data.expire_set_er', _dt_to_str),
            Extractor('partner_data_station_from_title', 'current_partner_data.station_from_title'),
            Extractor('partner_data_station_to_title', 'current_partner_data.station_to_title'),
            Extractor('partner_data_start_station_title', 'current_partner_data.start_station_title'),
            Extractor('partner_data_end_station_title', 'current_partner_data.end_station_title'),
            Extractor('partner_data_compartment_gender', 'current_partner_data.compartment_gender',
                      lambda v: GenderChoice(v).value),
            Extractor('partner_data_is_reservation_prolonged', 'current_partner_data.is_reservation_prolonged'),
            Extractor('partner_data_is_order_cancelled', 'current_partner_data.is_order_cancelled'),
            Extractor('partner_data_reservation_datetime', 'current_partner_data.reservation_datetime',
                      _dt_to_timestamp),
            Extractor('partner_data_reservation_datetime_str', 'current_partner_data.reservation_datetime', _dt_to_str),
            Extractor('partner_data_is_suburban', 'current_partner_data.is_suburban'),
            Extractor('partner_data_is_only_full_return_possible', 'current_partner_data.is_only_full_return_possible'),
            # source
            Extractor('source_req_id', 'source.req_id'),
            Extractor('source_device', 'source.device', lambda v: TrainPurchaseSource(v).value),
            Extractor('source_utm_source', 'source.utm_source'),
            Extractor('source_utm_medium', 'source.utm_medium'),
            Extractor('source_utm_campaign', 'source.utm_campaign'),
            Extractor('source_utm_term', 'source.utm_term'),
            Extractor('source_utm_content', 'source.utm_content'),
            Extractor('source_from', 'source.from_'),
            Extractor('source_gclid', 'source.gclid'),
            Extractor('source_terminal', 'source.terminal'),
            Extractor('source_partner', 'source.partner'),
            Extractor('source_subpartner', 'source.subpartner'),
            Extractor('source_partner_uid', 'source.partner_uid'),
            Extractor('source_test_id', 'source.test_id'),
            Extractor('source_test_buckets', 'source.test_buckets'),
            Extractor('source_icookie', 'source.icookie'),
            Extractor('source_serp_uuid', 'source.serp_uuid'),
            Extractor('source_is_transfer', 'source.is_transfer'),
            # station_ids
            Extractor('station_from_id', 'station_from_id'),
            Extractor('station_to_id', 'station_to_id'),
        ]
    )


class CpaOrderResponseSchema(object):
    """
    Преобразует заказ в словарь для CPA-платформы
    """
    EXTRACTORS = _create_extractors()

    @classmethod
    def flatten_order(cls, order):
        cls._prepare_order(order)
        return {extractor.name: extractor.get_value(order) for extractor in cls.EXTRACTORS}

    @staticmethod
    def _prepare_order(order):
        # TODO: эффективней вытаскивать зависимые объекты для нескольких заказов сразу. Возвраты, платежи, и т.п..
        # То есть сделать что-то вроде fetch_related.
        refunds = list(order.iter_refunds())
        order.done_refunds = [r for r in refunds if r.status in (RefundStatus.PARTNER_REFUND_DONE, RefundStatus.DONE)]
        refunded_blanks = {b for r in order.done_refunds for b in r.blank_ids}
        refunded_insurance_ids = {b for r in order.done_refunds for b in r.insurance_ids}
        order.refunded_tickets = [t for t in order.iter_tickets() if t.blank_id in refunded_blanks]
        order.refunded_insurances = [i for i in order.iter_insurances() if i.operation_id in refunded_insurance_ids]
        order.payment = order.current_billing_payment
