# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from datetime import timedelta

from bson import ObjectId
from marshmallow import Schema, fields
from mongoengine import Q
from rest_framework import status
from rest_framework.decorators import api_view
from rest_framework.pagination import LimitOffsetPagination
from rest_framework.response import Response

from travel.rasp.library.python.common23.date.environment import now_utc
from travel.rasp.train_api.train_purchase.core.models import TrainOrder, TrainRefund, RefundStatus
from travel.rasp.train_api.train_purchase.views.cpa.serialization import CpaOrderResponseSchema

PAGE_SIZE = 40
SHIFT_FROM_CREATION = timedelta(minutes=30)


@api_view(['GET'])
def orders_for_cpa_collector(request):
    """
    Ручка отдает заказы в плоской структуре (снапшоты) для CPA-платформы.
    Отдает заказы, от создания которых прошло не менее 30 минут.
    https://st.yandex-team.ru/TRAINS-2775
    """
    data, errors = CpaOrdersQuerySchema().load(request.GET)
    if errors:
        return Response({
            'errors': errors
        }, status=status.HTTP_400_BAD_REQUEST)

    orders_queryset = build_orders_queryset(data)

    paginator = LimitOffsetPagination()
    paginator.default_limit = PAGE_SIZE
    return paginator.get_paginated_response({
        'orders': [
            CpaOrderResponseSchema.flatten_order(order)
            for order in paginator.paginate_queryset(orders_queryset, request)
        ]
    })


@api_view(['GET'])
def orders_with_refunds_for_cpa_collector(request):
    """
    Ручка отдает заказы в плоской структуре (снапшоты) для CPA-платформы.
    Отдает заказы с успешными возвратами.
    https://st.yandex-team.ru/TRAINS-2775
    """
    data, errors = CpaOrdersQuerySchema().load(request.GET)
    if errors:
        return Response({
            'errors': errors
        }, status=status.HTTP_400_BAD_REQUEST)

    refunds_queryset = build_refunds_queryset(data)

    paginator = LimitOffsetPagination()
    paginator.default_limit = PAGE_SIZE

    order_uids = [r.order_uid for r in paginator.paginate_queryset(refunds_queryset, request)]

    return paginator.get_paginated_response({
        'orders': [
            CpaOrderResponseSchema.flatten_order(order)
            for order in TrainOrder.objects.filter(uid__in=order_uids)
        ]
    })


def build_orders_queryset(data):
    filters = Q()
    if 'utc_from' in data:
        filters &= Q(id__gte=ObjectId.from_datetime(data['utc_from']))
    dt_to = now_utc() - SHIFT_FROM_CREATION
    if 'utc_to' in data and data['utc_to'] < dt_to:
        dt_to = data['utc_to']
    filters &= Q(id__lt=ObjectId.from_datetime(dt_to))

    return TrainOrder.objects.filter(filters).exclude('process')


def build_refunds_queryset(data):
    filters = Q(status__in=(RefundStatus.PARTNER_REFUND_DONE, RefundStatus.DONE))
    if 'utc_from' in data:
        filters &= Q(created_at__gte=data['utc_from'])
    if 'utc_to' in data:
        filters &= Q(created_at__lt=data['utc_to'])

    return TrainRefund.objects.filter(filters).exclude('process')


class CpaOrdersQuerySchema(Schema):
    utc_from = fields.DateTime()
    utc_to = fields.DateTime()
