# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from django.conf import settings
from django.utils.text import force_text
from rest_framework import status
from rest_framework.response import Response
from rest_framework.routers import SimpleRouter
from rest_framework.viewsets import ViewSet
from ylog.context import log_context

from common.dev_tools.swagger import swagger_aware_view_set
from common.dynamic_settings.default import conf
from common.email_sender import guaranteed_send_email
from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from common.workflow.registry import run_process
from travel.rasp.train_api.train_partners.base.insurance.checkout import checkout
from travel.rasp.train_api.train_purchase.serialization import InsuranceCheckoutSchema
from travel.rasp.train_api.train_purchase.utils.decorators import order_detail_route
from travel.rasp.train_api.train_purchase.workflow.booking import TRAIN_BOOKING_PROCESS

log = logging.getLogger(__name__)

INSURANCE_PRICING_TIMEOUT_SEC = 15
INSURANCE_CHECKOUT_TIMEOUT_SEC = 30

define_setting('INSURANCE_ERROR_CAMPAIGN', {
    Configuration.PRODUCTION: 'WGJE1J13-ZWT1',
    Configuration.TESTING: '4B930J13-X4S1',
    Configuration.DEVELOPMENT: '4B930J13-X4S1'
}, default=None)


@swagger_aware_view_set
class OrderInsuranceViewSet(ViewSet):

    @order_detail_route()
    def checkout(self, request, order):
        """
        (Deprecated) Подтверждение бронирования, заказ страховки, переход к оплате
        ---
        deprecated: true
        parameters:
          - in: query
            name: query_params
            schema:
                $ref: 'InsuranceCheckoutSchema'
          - in: path
            name: pk
            required: true
            description: Uid заказа
            schema:
                type: string
        responses:
            200:
                description: ok
            409:
                description: страховки отключены
        """
        with log_context(order_uid=order.uid):
            try:
                if not order.insurance_enabled:
                    return Response({'errors': 'Insurance is not enabled in settings'},
                                    status=status.HTTP_409_CONFLICT)
                checkout_request, errors = InsuranceCheckoutSchema().load(request.query_params)
                if checkout_request['enabled']:
                    checkout(order, timeout=INSURANCE_CHECKOUT_TIMEOUT_SEC)
                run_process.apply_async([TRAIN_BOOKING_PROCESS, str(order.id), {'order_uid': order.uid}])
                order.update(**{'set__process__suspended': False})

                return Response({'ok': True})
            except Exception as error:
                log.exception(force_text(error))
                guaranteed_send_email(
                    key='insurance_error_email_{}'.format(order.uid),
                    to_email=conf.TRAIN_PURCHASE_ERRORS_EMAIL,
                    args={'order_uid': order.uid},
                    campaign=settings.INSURANCE_ERROR_CAMPAIGN,
                    log_context={'order_uid': order.uid},
                )
                return Response({'errors': 'Error in insurance/checkout'},
                                status=status.HTTP_500_INTERNAL_SERVER_ERROR)


orders_insurance_router = SimpleRouter()
orders_insurance_router.register('train-purchase/orders/insurance', OrderInsuranceViewSet, base_name='insurance')
