# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from marshmallow import Schema, fields, validates_schema, ValidationError
from marshmallow_enum import EnumField

from travel.rasp.train_api.train_purchase.core.enums import TrainPartner


class TicketFeeRequest(Schema):
    id = fields.String(missing=None)
    coach_type = fields.String(required=True)
    amount = fields.Decimal(required=True)
    service_amount = fields.Decimal(required=True)


class TicketFeeResponse(Schema):
    id = fields.String()
    coach_type = fields.String()
    amount = fields.Decimal()
    service_amount = fields.Decimal()
    fee = fields.Decimal()
    partner_fee = fields.Decimal()
    partner_refund_fee = fields.Decimal()


class TicketsFeeRequestSchema(Schema):
    yandex_uid = fields.String(missing=None)
    partner = EnumField(TrainPartner, by_value=True, missing=TrainPartner.IM)
    tickets = fields.Nested(TicketFeeRequest, many=True, required=True)


class TicketsFeeResponseSchema(Schema):
    yandex_uid = fields.String()
    partner = EnumField(TrainPartner, by_value=True)
    tickets = fields.Nested(TicketFeeResponse, many=True, required=True)


class OrderAuthDetailsQuerySchema(Schema):
    order_uid = fields.String(required=False, missing=None, load_from='trainOrderUid')
    express_order_number = fields.String(required=False, missing=None, load_from='expressOrderNumber')

    @validates_schema(skip_on_field_errors=False)
    def validate_order_data(self, data):
        if not data.get('order_uid') and not data.get('express_order_number'):
            raise ValidationError('Must be provided trainOrderUid or expressOrderNumber')


class OrderAuthDetailsResponseSchema(Schema):
    uid = fields.String(dump_to='trainOrderUid')
    express_order_number = fields.String(attribute='current_partner_data.order_num', dump_to='expressOrderNumber')
    phone = fields.String(attribute='user_info.phone')
    email = fields.String(attribute='user_info.email')
    passport_id = fields.String(attribute='user_info.uid', dump_to='passportId')
