# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

"""
Представления для портала путешествий
"""

from collections import namedtuple

from decimal import Decimal
from django.conf import settings
from rest_framework import status
from rest_framework.decorators import api_view
from rest_framework.response import Response

from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from travel.rasp.train_api.train_purchase.core.models import TrainOrder, ClientContracts
from travel.rasp.train_api.train_purchase.utils.fee_calculator import calculate_ticket_cost
from travel.rasp.train_api.train_purchase.utils.tvm import check_service_ticket, TvmServiceId
from travel.rasp.train_api.train_purchase.views.serialization import (
    OrderAuthDetailsResponseSchema, OrderAuthDetailsQuerySchema, TicketsFeeRequestSchema, TicketsFeeResponseSchema
)

define_setting('ORDER_AUTH_DETAILS_ALLOWED_SERVICE_IDS', {
    Configuration.PRODUCTION: [TvmServiceId.RASP_DEVELOPERS, TvmServiceId.HOTELS_PRODUCTION],
    Configuration.TESTING: [TvmServiceId.RASP_DEVELOPERS, TvmServiceId.HOTELS_TESTING],
}, default=[TvmServiceId.RASP_DEVELOPERS])


@api_view(['GET'])
@check_service_ticket(settings.ORDER_AUTH_DETAILS_ALLOWED_SERVICE_IDS)
def order_auth_details(request):
    """
    Возвращает данные из заказа для аутентификации и авторизации пользователя.
    Ожидает service ticket в заголовке.
    https://st.yandex-team.ru/TRAINS-2855
    ---
    parameters:
      - in: query
        name: query_params
        schema:
            $ref: 'OrderAuthDetailsQuerySchema'
    responses:
        200:
            description: данные для аутентификации и авторизации пользователя
            schema:
                properties:
                    result:
                        $ref: 'OrderAuthDetailsResponseSchema'
        400:
            description: не правильный запрос
        403:
            description: не прошла TVM авторизация
        404:
            description: не нашли нужный заказ
    """
    data, errors = OrderAuthDetailsQuerySchema().load(request.GET)
    if errors:
        return Response({
            'errors': errors
        }, status=status.HTTP_400_BAD_REQUEST)

    order = get_order(data)
    if not order:
        return Response({
            'errors': {'wrongRequest': ['no_such_order']}
        }, status=status.HTTP_404_NOT_FOUND)

    return Response({'result': OrderAuthDetailsResponseSchema().dump(order).data})


def get_order(data):
    try:
        if data.get('order_uid'):
            query = {'uid': data['order_uid']}
        else:
            query = {'partner_data_history__order_num': data['express_order_number']}
        query['removed__ne'] = True
        order = TrainOrder.objects.get(**query)
    except (TrainOrder.DoesNotExist, TrainOrder.MultipleObjectsReturned):
        return None
    return order


@api_view(['POST'])
def calculate_tickets_fee(request):
    """
    Рассчитать сборы яндекса по заказу
    ---
    parameters:
      - in: body
        name: post_schema
        schema:
            $ref: 'TicketsFeeRequestSchema'
    responses:
        201:
            description: билеты с комиссией
            schema:
                $ref: 'TicketsFeeResponseSchema'
        400:
            schema:
                properties:
                    errors:
                        type: object
        503:
            description: нет активных контрактов, покупка недоступна
    """
    request_data, errors = TicketsFeeRequestSchema().load(request.data)
    if errors:
        return Response({'errors': errors}, status=status.HTTP_400_BAD_REQUEST)

    contract = ClientContracts.get_active_contract(request_data['partner'])
    if contract is None:
        return Response({'errors': {'order': 'No active partner contract'}}, status=status.HTTP_503_SERVICE_UNAVAILABLE)

    class TicketData(namedtuple('TicketData', (
            'id',
            'coach_type',
            'amount',
            'service_amount',
            'fee',
            'partner_fee',
            'partner_refund_fee',
    ))):
        pass

    class OrderCost(namedtuple('OrderCost', (
            'partner',
            'tickets',  # список TicketData
            'yandex_uid',
    ))):
        pass

    result_tickets = []
    for t in request_data['tickets']:
        if not t['amount']:
            result_tickets.append(TicketData(
                t['id'], t['coach_type'], t['amount'], t['service_amount'], Decimal(0), Decimal(0), Decimal(0),
            ))
        else:
            ticket_cost = calculate_ticket_cost(
                contract, t['coach_type'], t['amount'], t['service_amount'], request_data['yandex_uid']
            )
            result_tickets.append(TicketData(
                t['id'], t['coach_type'], t['amount'], t['service_amount'], ticket_cost.full_fee,
                contract.partner_commission_sum, contract.partner_commission_sum2,
            ))

    result = OrderCost(request_data['partner'], result_tickets, request_data['yandex_uid'])

    return Response(TicketsFeeResponseSchema().dump(result).data, status=status.HTTP_200_OK)
