# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging
import time as c_time
from datetime import timedelta

import six
from django.conf import settings
from django.http.response import HttpResponse
from rest_framework import status
from rest_framework.decorators import api_view
from rest_framework.response import Response
from ylog.context import log_context

from common.dynamic_settings.default import conf
from common.settings.utils import define_setting
from common.utils import gen_hex_uuid
from travel.rasp.library.python.common23.date import environment
from common.utils.date import UTC_TZ
from common.utils.railway import get_railway_tz_by_point
from common.workflow.registry import run_process
from common.xgettext.i18n import gettext
from travel.rasp.train_api.helpers.rps_limiter import (
    rps_limiter, RPS_LIMITER_IM_SEARCH_KEY, RPS_LIMITER_IM_CARPRICING_KEY
)
from travel.rasp.train_api.tariffs.train.base.utils import get_point_express_code
from travel.rasp.train_api.train_bandit_api.logging import bandit_log_passenger_details
from travel.rasp.train_api.train_partners import get_partner_api
from travel.rasp.train_api.train_partners.base.electronic_registration import change_electronic_registration
from travel.rasp.train_api.train_partners.base.insurance.pricing import pricing
from travel.rasp.train_api.train_partners.base.prolong_reservation import prolong_reservation
from travel.rasp.train_api.train_partners.base.serialization import ScheduleQuerySchema, ScheduleSchema
from travel.rasp.train_api.train_partners.base.ticket_blank import download_ticket_blank, BlankFormat
from travel.rasp.train_api.train_partners.base.train_details.serialization import (
    TrainDetailsQuerySchema, TrainDetailsSchema, InternalTrainDetailsQuerySchema
)
from travel.rasp.train_api.train_partners.im.schedule import get_schedule_im
from travel.rasp.train_api.train_purchase.core.enums import OrderStatus
from travel.rasp.train_api.train_purchase.core.models import ClientContracts
from travel.rasp.train_api.train_purchase.serialization import (
    ChangeRegistrationSchema, DownloadQuerySchema, OrderSchema, ReserveTicketsSchema, CalculateRefundAmountSchema,
    UpdateTicketsStatusRequestSchema, LogBanditSchema
)
from travel.rasp.train_api.train_purchase.utils.decorators import order_view
from travel.rasp.train_api.train_purchase.utils.electronic_registration import (
    WrongBlankIdError, get_blank_ids_to_change_er, update_tickets_er_status, set_tickets_pending_status,
    RegistrationStatus
)
from travel.rasp.train_api.train_purchase.utils.order import (
    update_refund_amount, iter_refundable_tickets, update_tickets_statuses, UpdateTicketStatusMode
)
from travel.rasp.train_api.train_purchase.utils.order_tickets import (
    create_order, reserve_tickets, download_refund_blank, apply_personal_data
)
from travel.rasp.train_api.train_purchase.utils.sms_verification import verify_sms
from travel.rasp.train_api.train_purchase.workflow.booking import TRAIN_BOOKING_PROCESS

log = logging.getLogger(__name__)

REGISTRATION_SMS_ACTION_NAME = 'REGISTRATION_SMS'
MINIMUM_RESERVATION_MINUTES = 15
HTTP_422_UNPROCESSABLE_ENTITY = 422
define_setting('MAX_ORDER_QUERY_TIME', default=18)
define_setting('MIN_TIME_TO_INSURANCE_PRICING', default=3)


@api_view(['POST'])
def order_tickets(request):
    """
    Начать покупку
    ---
    parameters:
      - in: body
        name: post_schema
        schema:
            $ref: 'ReserveTicketsSchema'
    responses:
        201:
            description: Заказ
            schema:
                properties:
                    order:
                        $ref: 'OrderSchema'
        400:
            schema:
                properties:
                    errors:
                        type: object
        503:
            description: нет активных контрактов, покупка недоступна
    """
    start = c_time.time()
    order_data, errors = ReserveTicketsSchema().load(request.data)
    if errors:
        log.warning("Validation error occurred for order_tickets request: \n%s \nwith errors: \n%s",
                    request.data, errors)
        return Response({'errors': errors}, status=status.HTTP_400_BAD_REQUEST)

    contract = ClientContracts.get_active_contract(order_data['partner'])
    if contract is None:
        return Response({'errors': {'order': 'No active partner contract'}}, status=status.HTTP_503_SERVICE_UNAVAILABLE)

    order_uid = gen_hex_uuid()
    with log_context(order_uid=order_uid):
        reservation_result = reserve_tickets(order_data)
        order, personal_data = create_order(order_uid, order_data, reservation_result, contract)
        log.info('Создали заказ')

        if order_data.get('order_history', None):
            log.info('Действия пользователя до создания заказа:\n%s',
                     json.dumps(order_data['order_history'], indent=2, ensure_ascii=False, sort_keys=True))

        try_to_prolong_reservation(order)

        if not order.insurance_enabled:
            run_process.apply_async([TRAIN_BOOKING_PROCESS, str(order.id), {'order_uid': order.uid}])
        else:
            seconds_passed = c_time.time() - start
            time_to_pricing = settings.MAX_ORDER_QUERY_TIME - seconds_passed
            log.info('Осталось {} секунд на запрос цены страховки'.format(time_to_pricing))
            if time_to_pricing < settings.MIN_TIME_TO_INSURANCE_PRICING:
                log.info('Пропустили запрос цены по страховке')
            else:
                try:
                    pricing(order, timeout=time_to_pricing)
                    log.info('Запросили цены по страховке')
                except Exception:
                    log.exception('Не смогли запросить цены по страховке')

        apply_personal_data(order, personal_data)
        return Response({
            'order': OrderSchema().dump(order).data
        }, status=status.HTTP_201_CREATED)


def try_to_prolong_reservation(order):
    if not order.current_partner_data.is_three_hours_reservation_available:
        return

    if conf.TRAIN_PURCHASE_PROLONG_RESERVATION_MINUTES <= MINIMUM_RESERVATION_MINUTES:
        return

    try:
        real_reserved_to = prolong_reservation(order)
    except Exception:
        log.exception('Ошибка при продлении бронирования')
        return

    reserved_to = min(
        real_reserved_to,
        environment.now_aware() + timedelta(minutes=conf.TRAIN_PURCHASE_PROLONG_RESERVATION_MINUTES)
    )
    update_spec = dict(
        set__reserved_to=reserved_to.astimezone(UTC_TZ).replace(tzinfo=None),
    )
    update_spec.update({
        'set__{}__is_reservation_prolonged'.format(order.current_partner_data_lookup_name): True
    })
    order.modify(**update_spec)


@api_view(['GET'])
@order_view()
def download_blank(request, order):
    """
    Скачать бланк билета
    ---
    parameters:
      - in: query
        name: query_params
        schema:
            $ref: 'DownloadQuerySchema'
      - in: path
        name: uid
        required: true
        description: Uid заказа
        schema:
            type: string
    responses:
        200:
            description: бланк билета в запрошенном формате pdf/html
        400:
            schema:
                properties:
                    errors:
                        type: object
        409:
            description: статус заказа не подходит для выполения операции
    """
    if order.status != OrderStatus.DONE:
        return Response({
            'errors': {'order': 'Can not download blank in status "{}"'.format(order.status)},
        }, status=status.HTTP_409_CONFLICT)

    query, errors = DownloadQuerySchema().load(request.GET)
    if errors:
        return Response(
            {'errors': errors},
            status=status.HTTP_400_BAD_REQUEST
        )

    ticket_format = query['ticket_format']
    if 'blank_id' in query:
        blank_id_to_ticket = {t.blank_id: t for t in order.iter_tickets()}
        ticket = blank_id_to_ticket.get(query['blank_id'])
        if not ticket:
            return Response({
                'errors': {'blank_id': 'Wrong blankId, ticket not found'}
            }, status=status.HTTP_404_NOT_FOUND)

        if ticket.refund:
            if not ticket.refund.blank_id:
                return Response(
                    {'errors': {'blank_id': 'No blank with such blankId'}},
                    status=status.HTTP_404_NOT_FOUND
                )
            refund_blank = download_refund_blank(order, ticket)
            return HttpResponse(refund_blank.content, content_type=BlankFormat.PDF)

    operation_id = order.current_partner_data.operation_id
    ticket_blank = download_ticket_blank(order, operation_id, ticket_format)
    return HttpResponse(ticket_blank, content_type=ticket_format.mime_type)


@api_view(['POST'])
@order_view()
def calculate_refund_amount(request, order):
    """
    Посчитать сумму к возврату
    ---
    parameters:
      - in: body
        name: post_schema
        schema:
            $ref: 'CalculateRefundAmountSchema'
      - in: path
        name: uid
        required: true
        description: Uid заказа
        schema:
            type: string
    responses:
        200:
            description: Заказ
            schema:
                properties:
                    order:
                        $ref: 'OrderSchema'
        400:
            schema:
                properties:
                    errors:
                        type: object
    """
    if not order.partner.is_active:
        return Response({
            'errors': 'Partner is not active',
        }, status=status.HTTP_400_BAD_REQUEST)

    data, errors = CalculateRefundAmountSchema().load(request.data)
    if errors:
        return Response({
            'errors': errors,
        }, status=status.HTTP_400_BAD_REQUEST)

    blank_ids = data['blank_ids']
    non_refundable_blank_ids = set(blank_ids) - {t.blank_id for t in iter_refundable_tickets(order)}
    if non_refundable_blank_ids:
        return Response({
            'errors': 'Non refundable blank ids {}'.format(non_refundable_blank_ids),
        }, status=status.HTTP_400_BAD_REQUEST)

    update_refund_amount(order, blank_ids)

    return Response({
        'order': OrderSchema().dump(order).data
    })


@api_view(['GET'])
@order_view()
def update_tickets_status(request, order):
    """
    Обновить статусы билетов из партнера
    ---
    parameters:
      - in: path
        name: uid
        required: true
        description: Uid заказа
        schema:
            type: string
    responses:
        200:
            description: Заказ
            schema:
                properties:
                    order:
                        $ref: 'OrderSchema'
        422:
            description: бронь отменена
    """
    query, errors = UpdateTicketsStatusRequestSchema().load(request.GET)
    if errors:
        return Response({
            'errors': errors,
        }, status=status.HTTP_400_BAD_REQUEST)

    if order.current_partner_data.is_order_cancelled:
        update_tickets_statuses(order, mode=UpdateTicketStatusMode.SIMPLE, set_personal_data=True,
                                set_order_warnings=True, first_actual_warning_only=query['first_actual_warning_only'])
        return Response({
            'order': OrderSchema().dump(order).data
        }, status=HTTP_422_UNPROCESSABLE_ENTITY)
    update_tickets_statuses(order, mode=UpdateTicketStatusMode.UPDATE_FROM_EXPRESS, set_personal_data=True,
                            set_order_warnings=True, first_actual_warning_only=query['first_actual_warning_only'])
    return Response({
        'order': OrderSchema().dump(order).data
    })


@api_view(['GET'])
@order_view()
def change_registration_status(request, order):
    """
    Изменить статус ЭР
    :type order: train_api.train_purchase.core.models.TrainOrder
    ---
    parameters:
      - in: query
        name: query_params
        schema:
            $ref: 'ChangeRegistrationSchema'
      - in: path
        name: uid
        required: true
        description: Uid заказа
        schema:
            type: string
    responses:
        200:
            description: Заказ
            schema:
                properties:
                    order:
                        $ref: 'OrderSchema'
        400:
            schema:
                properties:
                    errors:
                        type: object
    """
    if not order.partner.is_active:
        return Response({
            'errors': 'Partner is not active',
        }, status=status.HTTP_400_BAD_REQUEST)

    data, errors = ChangeRegistrationSchema().load(request.query_params)
    if errors:
        return Response({
            'errors': errors,
        }, status=status.HTTP_400_BAD_REQUEST)

    if order.current_partner_data.expire_set_er < environment.now_utc():
        return Response({
            'errors': {'order': 'Too late to change ER'},
        }, status=status.HTTP_400_BAD_REQUEST)

    blank_ids, new_status = data['blank_ids'], data['new_status']
    try:
        blank_ids_to_change = get_blank_ids_to_change_er(order, blank_ids, new_status)
    except WrongBlankIdError as e:
        return Response({
            'errors': {'blankIds': six.text_type(e)}
        }, status.HTTP_400_BAD_REQUEST)

    if data['new_status'] is RegistrationStatus.DISABLED:
        message_template = gettext(
            '{{code}} — код для отмены электронной регистрации на поезд по заказу № {}'
        ).format(order.current_partner_data.order_num)
    else:
        message_template = gettext(
            '{{code}} — код для прохождения электронной регистрации на поезд по заказу № {}'
        ).format(order.current_partner_data.order_num)

    response = verify_sms(request_sms_verification=data.get('request_sms_verification'),
                          sms_verification_code=data.get('sms_verification_code'),
                          phone=order.user_info.phone, message_template=message_template,
                          action_name=generate_registration_sms_action_name(data['blank_ids']),
                          action_data={'uid': order.uid,
                                       'new_status': data['new_status'].name,
                                       'blank_ids': sorted(data['blank_ids'])})
    if response is not None:
        return response

    try:
        if change_electronic_registration(order, blank_ids_to_change, data['new_status']):
            update_tickets_er_status(order, blank_ids, new_status)
        else:
            set_tickets_pending_status(order, blank_ids)
    except Exception:
        set_tickets_pending_status(order, blank_ids)
        raise

    return Response({'order': OrderSchema().dump(order).data})


@api_view(['GET'])
def train_details_view(request):
    """
    Повагонная информация о поезде для покупки билета.
    Пример запроса:
    /ru/api/train-details/?number=070Ч&stationFrom=9610384&stationTo=9610189&when=2016-11-05T12:11
    ---
    parameters:
      - in: query
        name: query_params
        schema:
            $ref: 'TrainDetailsQuerySchema'
    responses:
        200:
            description: Повагонная информация о поезде для покупки билета
            schema:
                properties:
                    trainDetails:
                        $ref: 'TrainDetailsSchema'
        400:
            schema:
                properties:
                    errors:
                        type: object
    """
    query_params = TrainDetailsQuerySchema.prepare_query_from_headers(request.META, request.query_params)
    query, errors = TrainDetailsQuerySchema().load(query_params)
    if errors:
        return Response({'errors': errors}, status=status.HTTP_400_BAD_REQUEST)
    if conf.TRAIN_PURCHASE_IM_RPS_LIMITER_ENABLED and not query.mock_im:
        query_allowed = rps_limiter.check_query_allowed(RPS_LIMITER_IM_CARPRICING_KEY,
                                                        conf.TRAIN_PURCHASE_IM_CARPRICING_RPS_LIMIT,
                                                        conf.TRAIN_PURCHASE_IM_RPS_LIMITER_PERIOD)
        if query_allowed:
            rps_limiter.save_query_token(RPS_LIMITER_IM_SEARCH_KEY, RPS_LIMITER_IM_CARPRICING_KEY)
        else:
            return Response(status=status.HTTP_429_TOO_MANY_REQUESTS)
    partner_api = get_partner_api(query.partner)
    return Response({'trainDetails': TrainDetailsSchema().dump(partner_api.get_train_details(query)).data})


@api_view(['GET'])
def internal_train_details_view(request):
    """
    Повагонная информация о поезде для внутренних сервисов.
    Пример запроса:
    /ru/api/internal/train-details/?number=070Ч&stationFrom=2000003&stationTo=2000004&when=2016-11-05T12:11
    ---
    parameters:
      - in: query
        name: query_params
        schema:
            $ref: 'InternalTrainDetailsQuerySchema'
    responses:
        200:
            description: Повагонная информация о поезде
            schema:
                properties:
                    trainDetails:
                        $ref: 'TrainDetailsSchema'
        400:
            schema:
                properties:
                    errors:
                        type: object
    """
    query, errors = InternalTrainDetailsQuerySchema().load(request.query_params)
    if errors:
        return Response({'errors': errors}, status=status.HTTP_400_BAD_REQUEST)

    partner_api = get_partner_api(query.partner)
    return Response({'trainDetails': TrainDetailsSchema().dump(partner_api.get_train_details(query)).data})


@api_view(['GET'])
def schedule(request):
    """
    Расписание поездов от партнера. Сейчас из него берутся только времена начала продажи билетов
    Пример запроса:
    /ru/api/partner-schedule/?pointFrom=c213&pointTo=c2&when=2016-11-05T12:11
    ---
    parameters:
      - in: query
        name: query_params
        schema:
            $ref: 'ScheduleQuerySchema'
    responses:
        200:
            description: Расписание поездов от партнера
            schema:
                properties:
                    schedule:
                        $ref: 'ScheduleSchema'
        400:
            schema:
                properties:
                    errors:
                        type: object
    """
    query, errors = ScheduleQuerySchema().load(request.query_params)
    if errors:
        return Response({'errors': errors}, status=status.HTTP_400_BAD_REQUEST)

    point_from_code = get_point_express_code(query['point_from'])
    point_to_code = get_point_express_code(query['point_to'])

    if not (point_from_code and point_to_code):
        return Response({'errors': 'Невозможно выполнить запрос. Не найден код экспресс.'},
                        status=status.HTTP_400_BAD_REQUEST)

    schedule_im = get_schedule_im(
        point_from_code,
        point_to_code,
        query['when'],
        query['time_from'],
        query['time_to'],
        get_railway_tz_by_point(query['point_from']))

    return Response({'schedule': ScheduleSchema().dump(schedule_im).data})


@api_view(['POST'])
def log_bandit(request):
    """
    Сделать запись в лог Бандита
    ---
    parameters:
      - in: body
        name: post_schema
        schema:
            $ref: 'LogBanditSchema'
    responses:
        200:
            description: запись произведена
        400:
            description: некорректный запрос
        500:
            description: внутренняя ошибка
    """
    try:
        if conf.TRAIN_PURCHASE_BANDIT_LOGGING:
            query_params = LogBanditSchema.prepare_query_from_headers(request.META, request.GET)
            data, errors = LogBanditSchema().load(request.data)
            if errors:
                return Response({'errors': errors}, status=status.HTTP_400_BAD_REQUEST)

            data.update(query_params.dict())

            bandit_log_passenger_details(data)
    except Exception as e:
        log.exception('Error during log_bandit')
        return Response({'errors': e.message}, status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    return Response(status=status.HTTP_200_OK)


def generate_registration_sms_action_name(blank_ids):
    if blank_ids:
        return '{}-{}'.format(REGISTRATION_SMS_ACTION_NAME, '-'.join(sorted(blank_ids)))
    else:
        return REGISTRATION_SMS_ACTION_NAME
