# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from django.conf import settings

from common.workflow.sleep import SleepEvents, SleepTill, WaitAction, WaitTillStatuses
from travel.rasp.train_api.train_purchase.core.enums import OrderStatus
from travel.rasp.train_api.train_purchase.workflow.booking.cancel_order import CancelOrder, CancelOrderEvents
from travel.rasp.train_api.train_purchase.workflow.booking.cancel_payment import CancelPayment, CancelPaymentEvents
from travel.rasp.train_api.train_purchase.workflow.booking.check_insurance import CheckInsurance, CheckInsuranceEvents
from travel.rasp.train_api.train_purchase.workflow.booking.check_order import CheckOrder, CheckOrderEvents
from travel.rasp.train_api.train_purchase.workflow.booking.checkout_insurance import CheckoutInsurance, CheckoutInsuranceEvents
from travel.rasp.train_api.train_purchase.workflow.booking.confirm_order import ConfirmOrder, ConfirmOrderEvents
from travel.rasp.train_api.train_purchase.workflow.booking.create_insurance_refund import CreateInsuranceRefund
from travel.rasp.train_api.train_purchase.workflow.booking.create_payment import (
    CreatePayment, CreatePaymentEvents
)
from travel.rasp.train_api.train_purchase.workflow.booking.log_order import LogOrder, LogOrderEvents
from travel.rasp.train_api.train_purchase.workflow.booking.rebook_order import RebookOrder, RebookOrderEvents
from travel.rasp.train_api.train_purchase.workflow.booking.send_tickets_email import SendTicketsEmail, SendTicketsEmailEvents
from travel.rasp.train_api.train_purchase.workflow.booking.send_tickets_sms import SendTicketsSMS, SendTicketsSMSEvents
from travel.rasp.train_api.train_purchase.workflow.booking.set_insurance_status import SetInsuranceStatus, SetInsuranceStatusEvents
from travel.rasp.train_api.train_purchase.workflow.booking.set_pending import SetMaxPendingTill, SetMaxPendingTillEvents
from travel.rasp.train_api.train_purchase.workflow.booking.update_expired_order import (
    UpdateExpiredOrder, UpdateExpiredOrderStatuses, WAITING_EXPIRED_ORDER_INTERVAL
)
from travel.rasp.train_api.train_purchase.workflow.booking.update_order_status import UpdateOrderStatus, UpdateOrderStatusEvents
from travel.rasp.train_api.train_purchase.workflow.booking.update_order_tickets import UpdateOrderTickets, UpdateTicketStatuses
from travel.rasp.train_api.train_purchase.workflow.user_events import TrainBookingUserEvents

TRAIN_BOOKING_PROCESS = 'train_booking_process'


class OrderState(object):
    REBOOKING = 'rebooking'
    SET_INSURANCE_STATUS = 'set_insurance_status'
    CHECKOUT_INSURANCE = 'checkout_insurance'
    CHECK_INSURANCE = 'check_insurance'
    CREATE_PAYMENT = 'create_payment'
    CANCELLING = 'cancelling'
    CANCELLED = 'cancelled'
    WAITING_PAYMENT = 'waiting_payment'
    CONFIRMING = 'confirming'
    UPDATING_ORDER_TICKETS = 'updating_order_tickets'  # TODO: remove this state
    SET_MAX_PENDING_TILL = 'set_max_pending_till'
    WAITING_ORDER_CONFIRMATION = 'waiting_order_confirmation'
    CHECKING_ORDER_CONFIRMATION_AND_UPDATING = 'checking_order_confirmation_and_updating'
    SENDING_TICKETS_EMAIL = 'sending_tickets_email'
    CREATE_INSURANCE_REFUND = 'create_insurance_refund'
    SENDING_TICKETS_SMS = 'sending_tickets_sms'
    ABORTING = 'aborting'
    DONE = 'done'
    FAILED = 'failed'
    ORDER_EXPIRED = 'order_expired'
    WAITING_EXPIRED_ORDER = 'waiting_expired_order'
    CANCEL_EXPIRED_ORDER = 'cancel_expired_order'
    LOGGING_ORDER = 'logging_order'
    SET_STATUS_PAID = 'set_status_paid'
    SET_STATUS_PAYMENT_FAILED = 'set_status_payment_failed'
    SET_STATUS_CANCELLED = 'set_status_cancelled'


train_booking_scheme = {
    'initial_state': OrderState.REBOOKING,
    'states': {
        OrderState.REBOOKING: {
            'do': RebookOrder,
            'transitions': {
                RebookOrderEvents.DONE: OrderState.SET_INSURANCE_STATUS,
                RebookOrderEvents.SKIPPED: OrderState.SET_INSURANCE_STATUS,
                RebookOrderEvents.FAILED: OrderState.FAILED,
            },
        },
        OrderState.SET_INSURANCE_STATUS: {
            'do': SetInsuranceStatus,
            'transitions': {
                SetInsuranceStatusEvents.DONE: OrderState.CHECKOUT_INSURANCE,
                SetInsuranceStatusEvents.SKIPPED: OrderState.CREATE_PAYMENT,
                SetInsuranceStatusEvents.FAILED: OrderState.CREATE_PAYMENT,
            },
        },
        OrderState.CHECKOUT_INSURANCE: {
            'do': CheckoutInsurance,
            'transitions': {
                CheckoutInsuranceEvents.DONE: OrderState.CHECK_INSURANCE,
                CheckoutInsuranceEvents.FAILED: OrderState.CHECK_INSURANCE,
            },
        },
        OrderState.CHECK_INSURANCE: {
            'do': CheckInsurance,
            'transitions': {
                CheckInsuranceEvents.OK: OrderState.CREATE_PAYMENT,
                CheckInsuranceEvents.FAILED: OrderState.FAILED,
            },
        },
        OrderState.CREATE_PAYMENT: {
            'do': CreatePayment,
            'transitions': {
                TrainBookingUserEvents.ORDER_EXPIRE: OrderState.ORDER_EXPIRED,
                TrainBookingUserEvents.PAYMENT_CANCELLED: OrderState.SET_STATUS_CANCELLED,
                CreatePaymentEvents.OK: OrderState.WAITING_PAYMENT,
                CreatePaymentEvents.ORDER_CANCELLED: OrderState.FAILED,
                CreatePaymentEvents.PAYMENT_ALREADY_STARTED: OrderState.WAITING_PAYMENT
            },
        },
        OrderState.CANCELLING: {
            'do': CancelOrder,
            'transitions': {
                CancelOrderEvents.OK: OrderState.CANCELLED,
            }
        },
        OrderState.WAITING_PAYMENT: {
            'transitions': {
                TrainBookingUserEvents.PAYMENT_CANCELLED: OrderState.SET_STATUS_CANCELLED,
                TrainBookingUserEvents.PAID: OrderState.SET_STATUS_PAID,
                TrainBookingUserEvents.PAYMENT_FAILED: OrderState.SET_STATUS_PAYMENT_FAILED,
                TrainBookingUserEvents.REBOOKING: OrderState.REBOOKING,
            },
        },
        OrderState.SET_STATUS_CANCELLED: {
            'do': {'action': UpdateOrderStatus, 'args': (OrderStatus.CANCELLED,)},
            'transitions': {
                UpdateOrderStatusEvents.OK: OrderState.CANCELLING,
            }
        },
        OrderState.SET_STATUS_PAID: {
            'do': {'action': UpdateOrderStatus, 'args': (OrderStatus.PAID,)},
            'transitions': {
                UpdateOrderStatusEvents.OK: OrderState.SET_MAX_PENDING_TILL,
            }
        },
        OrderState.SET_STATUS_PAYMENT_FAILED: {
            'do': {'action': UpdateOrderStatus, 'args': (OrderStatus.PAYMENT_FAILED,)},
            'transitions': {
                UpdateOrderStatusEvents.OK: OrderState.FAILED,
            }
        },
        OrderState.SET_MAX_PENDING_TILL: {
            'do': SetMaxPendingTill,
            'transitions': {
                SetMaxPendingTillEvents.OK: OrderState.CONFIRMING,
                SetMaxPendingTillEvents.RETRY: OrderState.CHECKING_ORDER_CONFIRMATION_AND_UPDATING
            },
        },
        OrderState.CONFIRMING: {
            'do': ConfirmOrder,
            'transitions': {
                ConfirmOrderEvents.OK: OrderState.CHECKING_ORDER_CONFIRMATION_AND_UPDATING,
                ConfirmOrderEvents.IM_FAILED_NO_RETRY: OrderState.ABORTING,
                ConfirmOrderEvents.FAILED: OrderState.WAITING_ORDER_CONFIRMATION,
                'unreachable': OrderState.UPDATING_ORDER_TICKETS,  # TODO: remove this
            },
        },
        OrderState.UPDATING_ORDER_TICKETS: {
            'do': UpdateOrderTickets,
            'transitions': {
                UpdateTicketStatuses.FAILED: OrderState.ABORTING,
                UpdateTicketStatuses.DONE: OrderState.SENDING_TICKETS_EMAIL,
            },
        },
        OrderState.WAITING_ORDER_CONFIRMATION: {
            'do': {
                'action': SleepTill,
                'args': ('max_pending_till',)
            },
            'transitions': {
                SleepEvents.OK: OrderState.SET_MAX_PENDING_TILL,
            },
        },
        OrderState.CHECKING_ORDER_CONFIRMATION_AND_UPDATING: {
            'do': CheckOrder,
            'transitions': {
                CheckOrderEvents.PENDING: OrderState.WAITING_ORDER_CONFIRMATION,
                CheckOrderEvents.FAILED: OrderState.ABORTING,
                CheckOrderEvents.DONE: OrderState.SENDING_TICKETS_EMAIL,
                CheckOrderEvents.INSURANCE_FAILED: OrderState.CREATE_INSURANCE_REFUND,
            },
        },
        OrderState.CREATE_INSURANCE_REFUND: {
            'do': CreateInsuranceRefund,
            'transitions': {
                WaitTillStatuses.OK: OrderState.SENDING_TICKETS_EMAIL,
            }
        },
        OrderState.SENDING_TICKETS_EMAIL: {
            'do': SendTicketsEmail,
            'transitions': {
                SendTicketsEmailEvents.OK: OrderState.SENDING_TICKETS_SMS,
            },
        },
        OrderState.SENDING_TICKETS_SMS: {
            'do': SendTicketsSMS,
            'transitions': {
                SendTicketsSMSEvents.OK: OrderState.LOGGING_ORDER,
            }
        },
        OrderState.LOGGING_ORDER: {
            'do': LogOrder,
            'transitions': {
                LogOrderEvents.OK: OrderState.DONE,
            }
        },
        OrderState.ABORTING: {
            'do': CancelPayment,
            'transitions': {
                CancelPaymentEvents.OK: OrderState.FAILED,
            },
        },
        OrderState.DONE: {},
        OrderState.FAILED: {
            'transitions': {
                TrainBookingUserEvents.RETRY_PAYMENT: OrderState.CREATE_PAYMENT,
                TrainBookingUserEvents.ORDER_EXPIRE: OrderState.ORDER_EXPIRED,
            }
        },
        OrderState.ORDER_EXPIRED: {
            'do': UpdateExpiredOrder,
            'transitions': {
                UpdateExpiredOrderStatuses.OK: OrderState.FAILED,
                UpdateExpiredOrderStatuses.FAILED: OrderState.FAILED,
                UpdateExpiredOrderStatuses.CANCEL: OrderState.CANCEL_EXPIRED_ORDER,
                UpdateExpiredOrderStatuses.RETRY: OrderState.WAITING_EXPIRED_ORDER,
            },
        },
        OrderState.CANCEL_EXPIRED_ORDER: {
            'do': CancelOrder,
            'transitions': {
                CancelOrderEvents.OK: OrderState.FAILED,
            }
        },
        OrderState.WAITING_EXPIRED_ORDER: {
            'do': {'action': WaitAction, 'args': (WAITING_EXPIRED_ORDER_INTERVAL,)},
            'transitions': {
                WaitTillStatuses.OK: OrderState.ORDER_EXPIRED
            }
        },
        OrderState.CANCELLED: {},
    },
    'lock_alive_time': settings.WORKFLOW_LOCK_ALIVE_TIME,
    'lock_update_interval': settings.WORKFLOW_LOCK_UPDATE_INTERVAL_SECONDS
}
