# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from django.conf import settings
from requests import ConnectionError

from common.dynamic_settings.default import conf
from common.email_sender import guaranteed_send_email
from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from common.workflow.process import StateAction
from travel.rasp.train_api.train_partners.base.get_order_info import get_order_info
from travel.rasp.train_api.train_partners.im.base import ImError
from travel.rasp.train_api.train_purchase.core.enums import OrderStatus, OperationStatus, TravelOrderStatus, InsuranceStatus
from travel.rasp.train_api.train_purchase.utils.order import make_ticket_statuses_update, send_event_to_payment
from travel.rasp.train_api.train_purchase.workflow.base import make_partner_data_update
from travel.rasp.train_api.train_purchase.workflow.user_events import PaymentUserEvents

log = logging.getLogger(__name__)

CHECK_ORDER_COUNT = 5
CHECK_ORDER_DELAY_SECONDS = 3 * 60

define_setting('INSURANCE_CONFIRM_ERROR_CAMPAIGN', {
    Configuration.PRODUCTION: '0OIGPU43-H3P',
    Configuration.TESTING: '0K8BOU43-TN02',
}, default=None)


class CheckOrderEvents(object):
    PENDING = 'pending'
    FAILED = 'failed'
    DONE = 'done'
    INSURANCE_FAILED = 'insurance_failed'


class CheckOrder(StateAction):
    def do(self, data, *args, **kwargs):
        return self.check_order_im()

    def check_order_im(self):
        """
        https://st.yandex-team.ru/RASPFRONT-4054#1507727130000
        """
        partner_data = self.document.current_partner_data

        try:
            order_info = get_order_info(self.document)
        except (ImError, ConnectionError):
            log.exception('Ошибка при попытке обновить статус бронирования Заказа')
            return CheckOrderEvents.PENDING, {
                'inc__{}__check_transaction_counter'.format(self.document.current_partner_data_lookup_name): 1
            }

        is_last_try = partner_data.check_transaction_counter >= CHECK_ORDER_COUNT - 1
        if order_info.status == OperationStatus.OK:
            insurance_pending, insurance_failed, update_spec = self.check_insurance(order_info)
            if insurance_pending and is_last_try:
                insurance_pending = False
                guaranteed_send_email(
                    key='send_insurance_pending_email_{}'.format(self.document.uid),
                    to_email=conf.TRAIN_PURCHASE_ERRORS_EMAIL,
                    args={
                        'order_uid': self.document.uid,
                        'description': 'Статус страховок еще неизвестен, но продолжили покупку.'
                    },
                    campaign=settings.INSURANCE_CONFIRM_ERROR_CAMPAIGN,
                    log_context={'order_uid': self.document.uid},
                )
            if not insurance_pending:
                update_spec.update(make_partner_data_update(self.document, order_info))
                update_spec.update(make_ticket_statuses_update(self.document, order_info.tickets))
                update_spec.update({'set__status': OrderStatus.DONE, 'set__travel_status': TravelOrderStatus.DONE})
                if (self.document.insurance and self.document.insurance.status == InsuranceStatus.CHECKED_OUT
                        and not insurance_failed):
                    send_event_to_payment(self.document.current_billing_payment, PaymentUserEvents.CONFIRM_PAYMENT)
                return (CheckOrderEvents.INSURANCE_FAILED if insurance_failed else CheckOrderEvents.DONE), update_spec

        if order_info.status == OperationStatus.FAILED:
            return CheckOrderEvents.FAILED, {
                'set__status': OrderStatus.CONFIRM_FAILED,
                'set__travel_status': TravelOrderStatus.CANCELLED,
            }

        if is_last_try:
            return CheckOrderEvents.FAILED, {
                'inc__{}__check_transaction_counter'.format(self.document.current_partner_data_lookup_name): 1,
                'set__status': OrderStatus.CONFIRM_FAILED,
                'set__travel_status': TravelOrderStatus.CANCELLED,
            }
        else:
            return CheckOrderEvents.PENDING, {
                'inc__{}__check_transaction_counter'.format(self.document.current_partner_data_lookup_name): 1
            }

    def check_insurance(self, order_info):
        order = self.document
        pending = False
        failed = False
        update_spec = {}
        insurances = {i.operation_id: i for i in order_info.insurances}
        for i, passenger in enumerate(order.passengers):
            if not (passenger.insurance and passenger.insurance.operation_id):
                continue
            if passenger.insurance.operation_id not in insurances:
                pending = True
                continue
            insurance_info = insurances[passenger.insurance.operation_id]
            update_spec['set__passengers__{}__insurance__operation_status'.format(i)] = insurance_info.operation_status
            if insurance_info.operation_status == OperationStatus.IN_PROCESS:
                pending = True
            elif insurance_info.operation_status == OperationStatus.FAILED:
                failed = True
        return pending, failed, update_spec
