# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import timedelta

from common.dynamic_settings.default import conf
from travel.rasp.library.python.common23.date.environment import now_utc
from common.workflow.process import StateAction
from travel.rasp.train_api.train_partners.base.get_order_info import get_order_info
from travel.rasp.train_api.train_purchase.core.enums import OrderStatus, RebookingStatus, InsuranceStatus
from travel.rasp.train_api.train_purchase.utils.order_tickets import re_reserve_tickets_and_check

log = logging.getLogger(__name__)


class RebookOrderEvents(object):
    DONE = 'done'
    SKIPPED = 'skipped'
    FAILED = 'failed'


class RebookOrder(StateAction):
    def do(self, data, *args, **kwargs):
        order = self.document

        status, update_spec = self.rebook_order(order)
        update_spec.update({
            'set__rebooking_info__status': status,
        })

        if status == RebookingStatus.DONE:
            return RebookOrderEvents.DONE, update_spec
        elif status in (RebookingStatus.DISABLED, RebookingStatus.SKIPPED):
            return RebookOrderEvents.SKIPPED, update_spec
        else:
            update_spec.update({
                'set__status': OrderStatus.CONFIRM_FAILED,
            })
            return RebookOrderEvents.FAILED, update_spec

    @classmethod
    def rebook_order(cls, order):
        update_spec = {}

        try:
            if not order.rebooking_info or not order.rebooking_info.enabled or not order.rebooking_info.cycle_until:
                return RebookingStatus.DISABLED, update_spec

            if order.rebooking_info.cycle_until > now_utc():
                return RebookingStatus.SKIPPED, update_spec

            cycle_number = (order.rebooking_info.cycle_number or 0) + 1
            if cycle_number > conf.TRAIN_PURCHASE_RESERVATION_MAX_CYCLES:
                return RebookingStatus.OVERLIMIT, update_spec

            if order.status == OrderStatus.DONE or getattr(order.process, 'state', None) == 'unhandled_exception_state'\
                    or getattr(order.current_billing_payment.process, 'state', None) == 'unhandled_exception_state':
                log.error('Cannot rebook due state of order {}'.format(order.uid))
                return RebookingStatus.INVALID_ORDER, update_spec

            order_info = get_order_info(order)

            success, reserve_update_spec, mismatch = re_reserve_tickets_and_check(
                order,
                order_info,
                cancel_required=True,
            )
            update_spec.update(reserve_update_spec)

            if not success:
                log.error('Cannot rebook due: {}'.format(mismatch))
                return RebookingStatus.MISMATCH, update_spec

            if order.insurance and order.insurance.status == InsuranceStatus.CHECKED_OUT:
                update_spec.update({
                    'set__insurance__status': InsuranceStatus.ACCEPTED
                })

            update_spec.update({
                'set__rebooking_info__cycle_number': cycle_number,
                'set__rebooking_info__cycle_until': now_utc() + timedelta(
                    minutes=conf.TRAIN_PURCHASE_RESERVATION_PARTNER_TIMEOUT),
            })

            return RebookingStatus.DONE, update_spec

        except Exception:
            log.exception('Exception during rebook_order')
            return RebookingStatus.FAILED, update_spec
