# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import timedelta

from django.conf import settings

from common.dynamic_settings.default import conf
from common.email_sender import guaranteed_send_email
from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from travel.rasp.library.python.common23.date.environment import now_utc
from common.workflow.process import StateAction
from travel.rasp.train_api.train_partners.base import RzhdStatus
from travel.rasp.train_api.train_partners.base.get_order_info import get_order_info
from travel.rasp.train_api.train_purchase.core.enums import OperationStatus, OrderStatus
from travel.rasp.train_api.train_purchase.utils.order import make_ticket_statuses_update, make_order_status_update

log = logging.getLogger(__name__)


MAX_RETRY_UPDATE_STATUS_PERIOD = timedelta(hours=24)
WAITING_EXPIRED_ORDER_INTERVAL = timedelta(hours=1)
PAID_ORDER_STATUSES = [OrderStatus.PAID, OrderStatus.DONE]
ERROR_MESSAGE_PARTNER_STATUS_DONE = 'Заказ был выкуплен у партнера'
define_setting('ORDER_CANCEL_ERRORS_CAMPAIGN', {
    Configuration.PRODUCTION: '0US3OBW2-EL12',
    Configuration.TESTING: 'WSKWNBW2-M5N1',
    Configuration.DEVELOPMENT: 'WSKWNBW2-M5N1'
}, default=None)


class UpdateExpiredOrderStatuses(object):
    OK = 'ok'
    RETRY = 'retry'
    FAILED = 'failed'
    CANCEL = 'cancel'


class UpdateExpiredOrder(StateAction):
    def do(self, data, *args, **kwargs):
        order = self.document
        try:
            order_info = get_order_info(order)
            update_spec = make_ticket_statuses_update(order, order_info.tickets)
            update_spec.update(make_order_status_update(order, order_info))
            if check_order_must_be_cancelled(order, order_info):
                log.info('order must be cancelled')
                return UpdateExpiredOrderStatuses.CANCEL, update_spec

            return UpdateExpiredOrderStatuses.OK, update_spec
        except Exception:
            log.exception('error in update tickets')
            if order.reserved_to + MAX_RETRY_UPDATE_STATUS_PERIOD < now_utc():
                return UpdateExpiredOrderStatuses.FAILED
            return UpdateExpiredOrderStatuses.RETRY


def check_order_must_be_cancelled(order, order_info):
    if order_is_paid(order):
        return False

    if tickets_are_confirmed(order_info):
        send_problem_order_cant_be_cancelled(order, ERROR_MESSAGE_PARTNER_STATUS_DONE)
        return False

    return not all_tickets_are_cancelled_or_refunded(order_info)


def order_is_paid(order):
    return order.status in PAID_ORDER_STATUSES


def tickets_are_confirmed(order_info):
    return order_info.status == OperationStatus.OK


def all_tickets_are_cancelled_or_refunded(order_info):
    return all(t.rzhd_status in [RzhdStatus.CANCELLED, RzhdStatus.REFUNDED] for t in order_info.tickets)


def send_problem_order_cant_be_cancelled(order, message):
    email = conf.TRAIN_PURCHASE_ERRORS_EMAIL
    args = {
        'order_uid': order.uid,
        'order_status': order.status.value,
        'error_message': message,
    }
    guaranteed_send_email(
        key='send_problem_order_cant_be_cancelled_{}'.format(order.uid),
        to_email=email,
        args=args,
        campaign=settings.ORDER_CANCEL_ERRORS_CAMPAIGN,
        log_context={'order_uid': order.uid},
    )
