# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from django.conf import settings

from common.workflow.sleep import Sleep, SleepEvents
from travel.rasp.train_api.train_purchase.workflow.payment.cancel_payment import CancelPayment, CancelPaymentEvents
from travel.rasp.train_api.train_purchase.workflow.payment.check_payment import (
    CHECK_PAYMENT_DELAY_SECONDS, CheckPayment, CheckPaymentEvents
)
from travel.rasp.train_api.train_purchase.workflow.payment.clear_payment import ClearPaymentEvents, ClearPayment
from travel.rasp.train_api.train_purchase.workflow.payment.create_payment import (
    CREATE_PAYMENT_DELAY_SECONDS, CreatePayment, CreatePaymentEvents
)
from travel.rasp.train_api.train_purchase.workflow.payment.create_refund_payment import (
    CreateRefundPayment, CreateRefundPaymentEvents
)
from travel.rasp.train_api.train_purchase.workflow.payment.rebook_order import RebookOrder, RebookOrderEvents
from travel.rasp.train_api.train_purchase.workflow.payment.refund_payment import RefundPaymentEvents, RefundPayment
from travel.rasp.train_api.train_purchase.workflow.payment.resize_payment import ResizePayment, ResizePaymentEvents
from travel.rasp.train_api.train_purchase.workflow.payment.sending_payment_event import (
    SendingPaymentEvent, SendingPaymentEventEvents
)
from travel.rasp.train_api.train_purchase.workflow.user_events import PaymentUserEvents, TrainBookingUserEvents

PAYMENT_PROCESS = 'payment_process'


class PaymentState(object):
    PAYING = 'paying'
    WAITING_CREATE_PAYMENT_RETRY = 'waiting_create_payment_retry'
    WAITING_PAYMENT = 'waiting_payment'
    REBOOKING = 'rebooking'
    CHECKING_PAYMENT = 'checking_payment'
    SENDING_PAYMENT_DONE = 'sending_payment_done'
    HOLDING = 'holding'
    CLEARING = 'clearing'
    UNHOLDING = 'unholding'
    CLEARED = 'cleared'
    SENDING_PAYMENT_FAILED = 'sending_payment_failed'
    FAILED = 'failed'
    CANCELLING = 'cancelling'
    SENDING_PAYMENT_CANCELLED = 'sending_payment_cancelled'
    CREATE_REFUND = 'create_refund'
    REFUNDING = 'refunding'
    RESIZING = 'resizing'


payment_scheme = {
    'initial_state': PaymentState.PAYING,
    'states': {
        PaymentState.PAYING: {
            'do': CreatePayment,
            'transitions': {
                PaymentUserEvents.CANCEL: PaymentState.CANCELLING,
                CreatePaymentEvents.OK: PaymentState.WAITING_PAYMENT,
                CreatePaymentEvents.NEED_RETRY: PaymentState.WAITING_CREATE_PAYMENT_RETRY,
                CreatePaymentEvents.FAILED: PaymentState.SENDING_PAYMENT_FAILED,
            },
        },
        PaymentState.SENDING_PAYMENT_FAILED: {
            'do': {'action': SendingPaymentEvent, 'args': (TrainBookingUserEvents.PAYMENT_FAILED,)},
            'transitions': {
                SendingPaymentEventEvents.OK: PaymentState.FAILED,
            }
        },
        PaymentState.WAITING_CREATE_PAYMENT_RETRY: {
            'do': {
                'action': Sleep,
                'args': (CREATE_PAYMENT_DELAY_SECONDS,)
            },
            'transitions': {
                PaymentUserEvents.CANCEL: PaymentState.CANCELLING,
                SleepEvents.OK: PaymentState.PAYING,
            },
        },
        PaymentState.WAITING_PAYMENT: {
            'do': {
                'action': Sleep,
                'args': (CHECK_PAYMENT_DELAY_SECONDS,)
            },
            'transitions': {
                PaymentUserEvents.CANCEL: PaymentState.CANCELLING,
                SleepEvents.OK: PaymentState.REBOOKING,
            },
        },
        PaymentState.REBOOKING: {
            'do': RebookOrder,
            'transitions': {
                RebookOrderEvents.DONE: PaymentState.CHECKING_PAYMENT,
                RebookOrderEvents.SKIPPED: PaymentState.CHECKING_PAYMENT,
            },
        },
        PaymentState.CHECKING_PAYMENT: {
            'do': CheckPayment,
            'transitions': {
                PaymentUserEvents.CANCEL: PaymentState.CANCELLING,
                CheckPaymentEvents.OK: PaymentState.SENDING_PAYMENT_DONE,
                CheckPaymentEvents.PENDING: PaymentState.WAITING_PAYMENT,
                CheckPaymentEvents.FAILED: PaymentState.SENDING_PAYMENT_FAILED
            },
        },
        PaymentState.SENDING_PAYMENT_DONE: {
            'do': {'action': SendingPaymentEvent, 'args': (TrainBookingUserEvents.PAID,)},
            'transitions': {
                SendingPaymentEventEvents.OK: PaymentState.HOLDING,
            },
        },
        PaymentState.HOLDING: {
            'transitions': {
                PaymentUserEvents.CONFIRM_PAYMENT: PaymentState.CLEARING,
                PaymentUserEvents.ABORT_PAYMENT: PaymentState.UNHOLDING,
                PaymentUserEvents.REFUND_PAYMENT: PaymentState.RESIZING,
            },
        },
        PaymentState.CLEARING: {
            'do': ClearPayment,
            'transitions': {
                ClearPaymentEvents.OK: PaymentState.CLEARED,
            }
        },
        PaymentState.UNHOLDING: {
            'do': CancelPayment,
            'transitions': {
                CancelPaymentEvents.OK: PaymentState.FAILED,
            },
        },
        PaymentState.CLEARED: {
            'transitions': {
                PaymentUserEvents.REFUND_PAYMENT: PaymentState.CREATE_REFUND,
            }
        },
        PaymentState.FAILED: {},
        PaymentState.CANCELLING: {
            'do': CancelPayment,
            'transitions': {
                CancelPaymentEvents.OK: PaymentState.SENDING_PAYMENT_CANCELLED,
            },
        },
        PaymentState.SENDING_PAYMENT_CANCELLED: {
            'do': {'action': SendingPaymentEvent, 'args': (TrainBookingUserEvents.PAYMENT_CANCELLED, True)},
            'transitions': {
                SendingPaymentEventEvents.OK: PaymentState.FAILED,
            }
        },
        PaymentState.CREATE_REFUND: {
            'do': CreateRefundPayment,
            'transitions': {
                CreateRefundPaymentEvents.CREATED: PaymentState.REFUNDING,
                CreateRefundPaymentEvents.FAILED: PaymentState.CLEARED,
            },
        },
        PaymentState.REFUNDING: {
            'do': RefundPayment,
            'transitions': {
                RefundPaymentEvents.DONE: PaymentState.CLEARED,
                RefundPaymentEvents.FAILED: PaymentState.CLEARED,
            },
        },
        PaymentState.RESIZING: {
            'do': ResizePayment,
            'transitions': {
                ResizePaymentEvents.OK: PaymentState.CLEARING,
            }
        }
    },
    'lock_alive_time': settings.WORKFLOW_LOCK_ALIVE_TIME,
    'lock_update_interval': settings.WORKFLOW_LOCK_UPDATE_INTERVAL_SECONDS
}
