# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from common.data_api.billing.trust_client import (
    TrustClient, TrustPaymentStatuses, TrustClientException, TrustClientRequestError, TRUST_HOSTS_BY_ENVIRONMENT,
    train_order_skip_trust
)
from common.settings.configuration import Configuration
from travel.rasp.library.python.common23.date import environment
from common.utils.try_hard import try_hard
from common.utils.yasmutil import YasmMetricSender, Metric
from common.workflow.process import StateAction
from travel.rasp.train_api.train_purchase.workflow.payment.create_payment import get_payment_timeout

CHECK_PAYMENT_DELAY_SECONDS = 1

log = logging.getLogger(__name__)


class CheckPaymentEvents(object):
    OK = 'ok'
    PENDING = 'pending'
    FAILED = 'failed'


class CheckPayment(StateAction):
    idempotent = True
    sender = YasmMetricSender(prefix='trust')

    def _send_payment_result_metric(self, name):
        # avvv - суффикс yasm, обозначающий среднее значение по хостам
        try:
            payment_time = environment.now_utc() - self.document.trust_created_at
            self.sender.send_many([
                Metric('{}_cnt'.format(name), 1, 'ammm'),
                Metric(
                    '{}_time'.format(name),
                    payment_time.total_seconds(),
                    'avvv'
                )
            ])
        except Exception:
            log.exception('Ошибка при попытке отправить метрики результата платежа')

    def do(self, _data, *_args, **_kwargs):
        payment = self.document
        order = payment.order

        if train_order_skip_trust():
            return CheckPaymentEvents.OK

        if get_payment_timeout(order.reserved_to) < 0:
            log.info('Время бронирования истекло')
            self._send_payment_result_metric('payment_outdated')
            return CheckPaymentEvents.FAILED

        try:
            payment_info = _get_payment_info(payment.purchase_token)
        except TrustClientException as ex:
            log.exception(repr(ex))
            self._send_payment_result_metric('payment_failed')
            return CheckPaymentEvents.FAILED

        if payment_info.status == TrustPaymentStatuses.STARTED:
            return CheckPaymentEvents.PENDING

        if payment_info.status == TrustPaymentStatuses.AUTHORIZED:
            # если мы используем продовый биллинг в тестинге, нужно сразу отменять платеж
            if payment.immediate_return:
                _unhold_production_payment(payment)
            self._send_payment_result_metric('payment_paid')
            return CheckPaymentEvents.OK, {
                'set__status': payment_info.status.value,
                'set__hold_at': environment.now_utc(),
            }
        else:
            self._send_payment_result_metric('payment_failed')
            return CheckPaymentEvents.FAILED, {
                'set__status': payment_info.status.value,
                'set__resp_code': payment_info.resp_code,
                'set__resp_desc': payment_info.resp_desc,
            }


@try_hard(retriable_exceptions=(TrustClientRequestError,))
def _get_payment_info(purchase_token):
    return TrustClient().get_payment_info(purchase_token)


@try_hard(retriable_exceptions=(TrustClientRequestError,))
def _unhold_production_payment(payment):
    trust_client = TrustClient(trust_host=TRUST_HOSTS_BY_ENVIRONMENT[Configuration.PRODUCTION])
    trust_client.unhold_payment(payment.purchase_token)
    log.info('Платеж отменен')
