# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from common.data_api.billing.trust_client import TrustClient, TrustClientRequestError, train_order_skip_trust
from travel.rasp.library.python.common23.date.environment import now_utc
from common.utils.try_hard import try_hard
from common.workflow.process import StateAction
from travel.rasp.train_api.train_purchase.core.models import RefundPaymentStatus

log = logging.getLogger(__name__)


class EmptyReversalIdError(Exception):
    pass


class ClearPaymentEvents(object):
    OK = 'ok'


class ClearPayment(StateAction):
    def do(self, _data, *_args, **_kwargs):
        if train_order_skip_trust():
            return ClearPaymentEvents.OK

        payment = self.document
        if payment.immediate_return:
            log.info('Пропускаем клиринг платежа, так как платеж отменен')
            return ClearPaymentEvents.OK
        if payment.clear_at:
            log.info('Пропускаем клиринг платежа, так как платеж заклирен')
            return ClearPaymentEvents.OK

        trust_client = TrustClient()

        @try_hard(retriable_exceptions=(TrustClientRequestError,))
        def clear_payment():
            trust_client.clear_payment(payment.purchase_token)

        clear_payment()
        payment_update_spec = {'set__clear_at': now_utc()}
        log.info('Успешно провели клиринг платежа.')

        # если это клиринг после ресайза, то нужно заполнить trust_reversal_id
        self._check_resize_status(trust_client, payment)

        return ClearPaymentEvents.OK, payment_update_spec

    def _check_resize_status(self, trust_client, payment):
        refund_payment = payment.current_refund_payment
        if not (refund_payment and refund_payment.payment_resized):
            return
        try:
            reversal_id = self._get_reversal_id(trust_client, refund_payment.purchase_token)
            refund_payment.refund_payment_status = RefundPaymentStatus.DONE
            refund_payment.trust_reversal_id = reversal_id
            refund_payment.save()
        except Exception:
            log.error('Не смогли обновить reversal_id для заказа %s и возврата %s',
                      refund_payment.order_uid, refund_payment.refund_uuid)

    @try_hard(retriable_exceptions=(TrustClientRequestError, EmptyReversalIdError), sleep_duration=2)
    def _get_reversal_id(self, trust_client, purchase_token):
        pay_info = trust_client.get_payment_info(purchase_token)
        if pay_info.reversal_id:
            return pay_info.reversal_id
        raise EmptyReversalIdError
