# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import timedelta

import pytz
from django.conf import settings

from common.data_api.billing.trust_client import (
    FiscalNdsType, TrustClient, TrustClientRequestError, TrustFiscalData, TrustPaymentOrder, train_order_skip_trust
)
from common.dynamic_settings.default import conf
from common.email_sender import guaranteed_send_email
from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from travel.rasp.library.python.common23.date import environment
from common.utils.exceptions import SimpleUnicodeException
from common.workflow.process import StateAction
from travel.rasp.train_api.train_purchase.core.enums import InsuranceStatus
from travel.rasp.train_api.train_purchase.core.models import TrainOrder
from travel.rasp.train_api.train_purchase.utils.billing import (
    BILLING_PARTNERS, BILLING_PRODUCT_SERVICE, BILLING_PRODUCT_TICKET, BILLING_PRODUCT_YANDEX_FEE,
    BILLING_PRODUCT_INSURANCE,
)

log = logging.getLogger(__name__)

PAYMENT_MARGIN_BEFORE_TICKET_RESERVATION_ENDS = timedelta(minutes=1)
CREATE_PAYMENT_COUNT = 5
CREATE_PAYMENT_DELAY_SECONDS = 5


define_setting('ORDER_ERROR_CAMPAIGN', {
    Configuration.PRODUCTION: 'GM6ZCR93-K3X1',
    Configuration.TESTING: 'C7J1CR93-RLI',
    Configuration.DEVELOPMENT: 'C7J1CR93-RLI'
}, default=None)


class CreatePaymentEvents(object):
    OK = 'ok'
    NEED_RETRY = 'need_retry'
    FAILED = 'failed'


class CreatePayment(StateAction):
    def do(self, _data, *_args, **_kwargs):
        if train_order_skip_trust():
            return CreatePaymentEvents.OK

        payment = self.document
        try:
            payment_update_spec = create_and_start_payment(payment.order)
        except TrustClientRequestError:
            if payment.create_payment_counter < CREATE_PAYMENT_COUNT:
                return CreatePaymentEvents.NEED_RETRY, {'inc__create_payment_counter': 1}
            else:
                return CreatePaymentEvents.FAILED
        else:
            payment_update_spec['set__trust_created_at'] = environment.now_utc()
            return CreatePaymentEvents.OK, payment_update_spec


def create_payment_orders(trust_client, order, ticket):
    partner = order.partner
    ticket_payment = ticket.payment
    ticket_order = service_order = fee_order = None
    ticket_amount = ticket_payment.amount
    billing_partner = BILLING_PARTNERS[partner]

    place_string = get_places_string(ticket.places)
    if ticket_amount:
        fiscal_inn = ticket.carrier_inn
        if not fiscal_inn:
            fiscal_inn = billing_partner.inn
            log.error('ИНН перевозчика не был задан в заказе %s.', order.uid)
            try:
                guaranteed_send_email(
                    key='send_long_external_event_email_{}'.format(order.uid),
                    to_email=conf.TRAIN_PURCHASE_ERRORS_EMAIL,
                    args={'order_uid': order.uid, 'description': 'ИНН перевозчика не был задан.'},
                    campaign=settings.ORDER_ERROR_CAMPAIGN,
                    log_context={'order_uid': order.uid},
                )
            except Exception:
                log.exception('Проблема с отправкой письма об отсутствии перевозчка в заказе %s', order.uid)

        service_amount = ticket_payment.service_amount

        if service_amount:
            ticket_amount -= service_amount
            service_order = TrustPaymentOrder(
                fiscal_nds=FiscalNdsType.from_rate(ticket_payment.service_vat.rate),
                fiscal_title='{}, {}'.format(BILLING_PRODUCT_SERVICE.name, place_string),
                fiscal_inn=fiscal_inn,
                order_id=trust_client.create_order(BILLING_PRODUCT_SERVICE.partner_to_id[partner]),
                price=service_amount,
            )

        ticket_order = TrustPaymentOrder(
            fiscal_nds=FiscalNdsType.from_rate(ticket_payment.tariff_vat and ticket_payment.tariff_vat.rate),
            fiscal_title='{}, {}'.format(BILLING_PRODUCT_TICKET.name, place_string),
            fiscal_inn=fiscal_inn,
            order_id=trust_client.create_order(BILLING_PRODUCT_TICKET.partner_to_id[partner]),
            price=ticket_amount,
        )

        fee_order = TrustPaymentOrder(
            fiscal_nds=FiscalNdsType.NDS_20,
            fiscal_title='{}, {}'.format(BILLING_PRODUCT_YANDEX_FEE.name, place_string),
            order_id=trust_client.create_order(BILLING_PRODUCT_YANDEX_FEE.partner_to_id[partner]),
            price=ticket_payment.fee,
        )
    return ticket_order, service_order, fee_order


def get_places_string(places):
    if not places:
        place_string = ''
    elif len(places) == 1:
        place_string = 'место {0}'.format(places[0])
    else:
        place_string = 'места {0}'.format(', '.join(places))
    return place_string


def create_payment_insurance(trust_client, insurance, partner, places):
    place_num = get_places_string(places) if places else 'без места'
    insurance_order = TrustPaymentOrder(
        fiscal_nds=FiscalNdsType.NDS_0,
        fiscal_title='{}, {}'.format(BILLING_PRODUCT_INSURANCE.name, place_num),
        fiscal_inn=BILLING_PARTNERS[partner].inn,
        order_id=trust_client.create_order(BILLING_PRODUCT_INSURANCE.partner_to_id[partner]),
        price=insurance.amount,
    )
    return insurance_order


def get_payment_timeout(utc_reserved_to):
    time_left = pytz.UTC.localize(utc_reserved_to) - environment.now_aware()
    payment_timeout = time_left - PAYMENT_MARGIN_BEFORE_TICKET_RESERVATION_ENDS
    return payment_timeout.total_seconds()


def create_and_start_payment(order):
    """
    :type order: common.apps.train_order.models.TrainOrder
    """
    trust_client = TrustClient(user_ip=order.user_info.ip, user_passport_uid=order.user_info.uid,
                               user_region_id=order.user_info.region_id)
    order_update_spec = {}
    order_find_spec = {'uid': order.uid}
    payment_update_spec = {}
    payment_orders = []
    insurance_status = order.insurance.status if order.insurance else None

    for ticket, ticket_key in order.iter_ticket_to_lookup_name():
        ticket_order, service_order, fee_order = create_payment_orders(trust_client, order, ticket)

        if ticket_order:
            order_update_spec['set__{}__payment__ticket_order_id'.format(ticket_key)] = ticket_order.order_id
            payment_orders.append(ticket_order)
            order_find_spec['{}__payment__ticket_order_id'.format(ticket_key)] = ticket.payment.ticket_order_id

        if service_order:
            order_update_spec['set__{}__payment__service_order_id'.format(ticket_key)] = service_order.order_id
            payment_orders.append(service_order)
            order_find_spec['{}__payment__service_order_id'.format(ticket_key)] = ticket.payment.service_order_id

        if fee_order:
            order_update_spec['set__{}__payment__fee_order_id'.format(ticket_key)] = fee_order.order_id
            payment_orders.append(fee_order)
            order_find_spec['{}__payment__fee_order_id'.format(ticket_key)] = ticket.payment.fee_order_id

    if insurance_status != InsuranceStatus.FAILED:
        for index, passenger in enumerate(order.passengers):
            insurance = getattr(passenger, 'insurance', None)
            if insurance and insurance.operation_id:
                insurance_order = create_payment_insurance(trust_client, insurance, order.partner,
                                                           passenger.tickets[0].places)
                trust_order_id_key = 'passengers__{}__insurance__trust_order_id'.format(index)
                order_update_spec['set__{}'.format(trust_order_id_key)] = insurance_order.order_id
                payment_orders.append(insurance_order)
                order_find_spec[trust_order_id_key] = insurance.trust_order_id

    billing_partner = BILLING_PARTNERS[order.partner]
    pass_params = {
        'terminal_route_data': {
            'description': order.source.terminal,
        },
    } if order.source and order.source.terminal else None
    purchase_token = trust_client.create_payment(
        payment_orders,
        get_payment_timeout(order.reserved_to),
        is_mobile=order.user_info.is_mobile,
        user_email=order.user_info.email,
        fiscal_data=TrustFiscalData(
            fiscal_partner_inn=billing_partner.inn,
            fiscal_partner_phone=billing_partner.phone,
            fiscal_taxation_type=billing_partner.taxation_type
        ),
        fiscal_title=order.get_fiscal_title(),
        pass_params=pass_params,
    )

    order_update_result = TrainOrder.objects.filter(**order_find_spec).update_one(**order_update_spec)
    if not order_update_result:
        raise SimpleUnicodeException('Не смогли обновить информацию о заказе')

    payment_url = trust_client.start_payment(purchase_token)
    payment_update_spec['set__purchase_token'] = purchase_token
    payment_update_spec['set__payment_url'] = payment_url

    log.info('Создали корзину и запустили ее на оплату. token %s url %s', purchase_token, payment_url)

    if conf.TRUST_USE_PRODUCTION_FOR_TESTING:
        payment_update_spec['set__immediate_return'] = True
    return payment_update_spec
