# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from collections import namedtuple

from common.data_api.billing.trust_client import (
    TrustClient, TrustRefundOrder, TrustClientRequestError, train_order_skip_trust
)
from travel.rasp.library.python.common23.date import environment
from common.utils.exceptions import SimpleUnicodeException
from common.utils.try_hard import try_hard
from common.workflow.process import StateAction
from travel.rasp.train_api.train_purchase.core.models import RefundPaymentStatus

log = logging.getLogger(__name__)

CHECK_PAYMENT_CLEARED_SLEEP_DURATION = 5
CHECK_PAYMENT_CLEARED_MAX_RETRIES = 3

REFUND_FAILED_MESSAGE = 'Не смогли вернуть деньги пользователю.'
REFUND_FAILED_MESSAGE_WITH_RETRY = 'Не смогли вернуть деньги пользователю. Попробуем повторить позже.'

RefundOrder = namedtuple('RefundOrder', ('order_id', 'result_amount', 'refund_amount'))


class WaitingForTrustPaymentCleared(SimpleUnicodeException):
    pass


class CreateRefundPaymentEvents(object):
    CREATED = 'created'
    FAILED = 'failed'


REFUND_EVENT_BY_STATUS = {
    RefundPaymentStatus.CREATED: CreateRefundPaymentEvents.CREATED,
    RefundPaymentStatus.DONE: CreateRefundPaymentEvents.CREATED,
    RefundPaymentStatus.FAILED: CreateRefundPaymentEvents.FAILED,
    RefundPaymentStatus.UNKNOWN: CreateRefundPaymentEvents.FAILED,
    RefundPaymentStatus.NEW: CreateRefundPaymentEvents.FAILED,
}


class CreateRefundPayment(StateAction):
    def do(self, _data, event_params, *_args, **_kwargs):
        payment = self.document
        payment.current_refund_payment_id = event_params
        refund_payment = payment.current_refund_payment
        payment_update_spec = {'current_refund_payment_id': event_params}

        if refund_payment.refund_payment_status != RefundPaymentStatus.NEW:
            return REFUND_EVENT_BY_STATUS[refund_payment.refund_payment_status], payment_update_spec

        refund_payment_status, trust_refund_id = do_refund_payment(payment, refund_payment)
        refund_payment_update_spec = {
            'set__refund_payment_status': refund_payment_status,
        }
        if trust_refund_id:
            refund_payment_update_spec['set__trust_refund_id'] = trust_refund_id
        if refund_payment_status == RefundPaymentStatus.FAILED:
            refund_payment_update_spec['set__refund_payment_finished_at'] = environment.now_utc()
        refund_payment.update(**refund_payment_update_spec)
        return REFUND_EVENT_BY_STATUS[refund_payment_status], payment_update_spec


def do_refund_payment(payment, refund_payment):
    trust_refund_id = None
    if train_order_skip_trust() or payment.immediate_return:
        return RefundPaymentStatus.DONE, trust_refund_id

    order = payment.order
    total_refund_amount = sum(t.refund.amount for t in order.iter_tickets()
                              if t.blank_id in refund_payment.refund_blank_ids and t.refund.amount)
    if total_refund_amount == 0 and not refund_payment.refund_insurance_ids:
        log.info('Нулевая сумма к возврату, пропускаем возврат денег пользователю.')
        return RefundPaymentStatus.DONE, trust_refund_id

    trust_client = TrustClient(user_ip=refund_payment.user_info.ip, user_region_id=refund_payment.user_info.region_id)

    try:
        check_payment_cleared(trust_client, order.current_billing_payment.purchase_token)
        trust_refund_id = create_refund_payment(trust_client, order, refund_payment)
    except WaitingForTrustPaymentCleared:
        log.exception(REFUND_FAILED_MESSAGE_WITH_RETRY)
        return RefundPaymentStatus.NEW, trust_refund_id
    except Exception:
        log.exception(REFUND_FAILED_MESSAGE)
        return RefundPaymentStatus.FAILED, trust_refund_id

    return RefundPaymentStatus.CREATED, trust_refund_id


@try_hard(retriable_exceptions=(TrustClientRequestError, WaitingForTrustPaymentCleared),
          max_retries=CHECK_PAYMENT_CLEARED_MAX_RETRIES,
          sleep_duration=CHECK_PAYMENT_CLEARED_SLEEP_DURATION)
def check_payment_cleared(trust_client, purchase_token):
    if not trust_client.is_payment_cleared(purchase_token):
        raise WaitingForTrustPaymentCleared('Клиринг платежа еще не прошел')


@try_hard(retriable_exceptions=(TrustClientRequestError,))
def create_refund_payment(trust_client, order, refund_payment):
    refund_orders = calculate_refund_payment_orders(order, refund_payment)
    trust_refund_orders = [TrustRefundOrder(order_id=r.order_id, delta_amount=r.refund_amount) for r in refund_orders]
    return trust_client.create_refund(trust_refund_orders, refund_payment.purchase_token)


def calculate_refund_payment_orders(order, refund_payment):
    refund_orders = []

    for ticket in order.iter_tickets():
        if ticket.blank_id in refund_payment.refund_blank_ids:
            if ticket.refund.amount:
                service_refund_amount = ticket.payment.service_amount
                ticket_refund_amount = ticket.refund.amount
                fee_refund_amount = ticket.refund.refund_yandex_fee_amount

                if service_refund_amount:
                    if ticket_refund_amount < service_refund_amount:
                        log.info('Не удается вернуть полную стоимость услуг. К возврату: {0}; услуги: {1}.'
                                 .format(ticket_refund_amount, service_refund_amount))
                        service_refund_amount = ticket_refund_amount
                    ticket_refund_amount -= service_refund_amount
                    refund_orders.append(RefundOrder(
                        order_id=ticket.payment.service_order_id,
                        refund_amount=service_refund_amount,
                        result_amount=ticket.payment.service_amount - service_refund_amount
                    ))
                if ticket_refund_amount:
                    refund_orders.append(RefundOrder(
                        order_id=ticket.payment.ticket_order_id,
                        refund_amount=ticket_refund_amount,
                        result_amount=ticket.payment.ticket_amount - ticket_refund_amount
                    ))
                if fee_refund_amount:
                    refund_orders.append(RefundOrder(
                        order_id=ticket.payment.fee_order_id,
                        refund_amount=fee_refund_amount,
                        result_amount=ticket.payment.fee - fee_refund_amount,
                    ))

    for insurance in order.iter_insurances():
        if insurance.operation_id in refund_payment.refund_insurance_ids:
            refund_orders.append(RefundOrder(
                order_id=insurance.trust_order_id,
                refund_amount=insurance.amount,
                result_amount=0,
            ))

    return refund_orders
