# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from common.data_api.billing.trust_client import (
    TrustClient, TrustRefundStatuses, TrustClientRequestError
)
from travel.rasp.library.python.common23.date import environment
from common.utils.exceptions import SimpleUnicodeException
from common.utils.try_hard import try_hard
from common.workflow.process import StateAction
from travel.rasp.train_api.train_purchase.core.models import RefundPaymentStatus

log = logging.getLogger(__name__)

CHECK_PAYMENT_REFUND_SLEEP_DURATION = 2
CHECK_PAYMENT_REFUND_MAX_RETRIES = 2

REFUND_FAILED_MESSAGE = 'Не смогли вернуть деньги пользователю.'
REFUND_FAILED_MESSAGE_WITH_RETRY = 'Не смогли вернуть деньги пользователю. Попробуем повторить позже.'
REFUND_UNKNOWN_MESSAGE = 'При возврате денег пользователю что-то пошло не так.'


class WaitingForTrustRefundStatus(SimpleUnicodeException):
    pass


class WaitingForTrustPaymentCleared(SimpleUnicodeException):
    pass


class RefundPaymentEvents(object):
    DONE = 'done'
    FAILED = 'failed'


REFUND_EVENT_BY_STATUS = {
    RefundPaymentStatus.UNKNOWN: RefundPaymentEvents.FAILED,
    RefundPaymentStatus.FAILED: RefundPaymentEvents.FAILED,
    RefundPaymentStatus.DONE: RefundPaymentEvents.DONE,
}


class RefundPayment(StateAction):
    def do(self, _data, *_args, **_kwargs):
        payment = self.document
        refund_payment = payment.current_refund_payment

        refund_payment_status = do_refund_payment(refund_payment)

        refund_payment_update_spec = {
            'set__refund_payment_status': refund_payment_status,
        }
        if refund_payment_status == RefundPaymentStatus.DONE:
            refund_payment_update_spec['set__refund_payment_finished_at'] = environment.now_utc()
        refund_payment.update(**refund_payment_update_spec)
        return REFUND_EVENT_BY_STATUS[refund_payment_status]


def do_refund_payment(refund_payment):
    try:
        if refund_payment.refund_payment_status == RefundPaymentStatus.DONE:
            log.info('Пропускаем возврат')
        else:
            trust_client = TrustClient(user_ip=refund_payment.user_info.ip,
                                       user_region_id=refund_payment.user_info.region_id)
            start_refund_payment(trust_client, refund_payment.trust_refund_id)
            check_refund_payment(trust_client, refund_payment.trust_refund_id)
    except (TrustClientRequestError, WaitingForTrustRefundStatus):
        log.exception(REFUND_UNKNOWN_MESSAGE)
        return RefundPaymentStatus.UNKNOWN
    except Exception:
        log.exception(REFUND_FAILED_MESSAGE)
        return RefundPaymentStatus.FAILED

    return RefundPaymentStatus.DONE


@try_hard(retriable_exceptions=(TrustClientRequestError,))
def start_refund_payment(trust_client, trust_refund_id):
    trust_client.start_refund(trust_refund_id)


@try_hard(max_retries=CHECK_PAYMENT_REFUND_MAX_RETRIES, sleep_duration=CHECK_PAYMENT_REFUND_SLEEP_DURATION,
          retriable_exceptions=(TrustClientRequestError, WaitingForTrustRefundStatus))
def check_refund_payment(trust_client, trust_refund_id):
    if trust_client.get_refund_status(trust_refund_id) != TrustRefundStatuses.SUCCESS:
        raise WaitingForTrustRefundStatus('Статус возврата в биллинге еще не SUCCESS.')
