# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from django.utils.functional import cached_property

from travel.rasp.train_api.train_partners.base.insurance.refund import insurance_refund, check_insurance_refund
from travel.rasp.train_api.train_partners.base.refund import (
    check_refund, make_refund, PartnerRefundFailed, PartnerRefundUnknown, PartnerRefundNotFound,
)

log = logging.getLogger(__name__)


class RefundManagerStates(object):
    DONE = 'done'
    FAILED = 'failed'
    NOT_FOUND = 'not_found'
    UNKNOWN = 'unknown'


class BaseRefundManager(object):
    def __init__(self):
        self.result = None
        self.state = None

    def do_refund(self):
        self._update_result(self._do_refund)

    def _do_refund(self):
        raise NotImplementedError

    def update_refund_info(self):
        self._update_result(self._update_refund_info)

    def _update_refund_info(self):
        raise NotImplementedError

    def update_refund_info_if_unknown(self):
        if self.state == RefundManagerStates.UNKNOWN:
            self.update_refund_info()

    def _update_result(self, func):
        try:
            self.result = func()
            self.state = RefundManagerStates.DONE
        except PartnerRefundNotFound:
            self.state = RefundManagerStates.NOT_FOUND
        except PartnerRefundUnknown:
            self.state = RefundManagerStates.UNKNOWN
        except PartnerRefundFailed:
            self.state = RefundManagerStates.FAILED


class BlankRefundManager(BaseRefundManager):
    def __init__(self, order, passenger, blank_id, refund_uuid):
        super(BlankRefundManager, self).__init__()
        self.order = order
        self.passenger = passenger
        self.blank_id = blank_id
        self.refund_uuid = refund_uuid

    @cached_property
    def reference_id(self):
        """Генерирую идентификатор транзакции для возврата."""
        blank_id = 'all' if self.blank_id is None else self.blank_id
        # для ИМ длина reference_id ограничена 32 символами, поэтому урежем
        return '{uid}:{blank_id}'.format(uid=self.refund_uuid, blank_id=blank_id)[-32:]

    def _do_refund(self):
        return make_refund(self.order, blank_id=self.blank_id, doc_id=self.passenger.doc_id,
                           reference_id=self.reference_id)

    def _update_refund_info(self):
        return check_refund(self.order, self.reference_id)


class InsuranceRefundManager(BaseRefundManager):
    def __init__(self, order, insurance_id, refund_uuid):
        super(InsuranceRefundManager, self).__init__()
        self.order = order
        self.insurance_id = insurance_id
        self.refund_uuid = refund_uuid

    @cached_property
    def reference_id(self):
        """Генерирую идентификатор транзакции для возврата."""
        # для ИМ длина reference_id ограничена 32 символами, поэтому урежем
        return '{uid}:{insurance_id}'.format(uid=self.refund_uuid, insurance_id=self.insurance_id)[-32:]

    def _do_refund(self):
        return insurance_refund(self.order, insurance_id=self.insurance_id, reference_id=self.reference_id)

    def _update_refund_info(self):
        return check_insurance_refund(self.order, self.reference_id)
