# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import timedelta
from time import sleep

from common.models.geo import Country, Station
from common.utils.try_hard import try_hard
from common.workflow.process import StateAction
from travel.rasp.train_api.train_partners.base import PartnerError, RzhdStatus
from travel.rasp.train_api.train_partners.base.get_order_info import get_order_info
from travel.rasp.train_api.train_purchase.core.enums import TrainPartner, OperationStatus, TravelOrderStatus
from travel.rasp.train_api.train_purchase.core.models import RefundStatus, ClientContracts, TrainOrder, TrainRefund
from travel.rasp.train_api.train_purchase.workflow.ticket_refund.refund_managers import (
    RefundManagerStates, BlankRefundManager, InsuranceRefundManager
)

log = logging.getLogger(__name__)

CHECK_ORDER_REFUND_PARAMS = {
    TrainPartner.IM: {
        'DELAY_BETWEEN_RETRIES': timedelta(minutes=3),
        'MAX_RETRIES': 5
    },
    TrainPartner.UFS: {
        'DELAY_BETWEEN_RETRIES': timedelta(minutes=5),
        'MAX_RETRIES': 3
    }
}


class RefundOrderEvents(object):
    DONE = 'done'
    OFFICE_REFUND = 'OFFICE_REFUND'
    NO_ACTIVE_CONTRACT = 'no_active_contract'
    FAILED = 'failed'


class RefundOrder(StateAction):
    def do(self, data, *_args, **_kwargs):
        refund = self.document
        if refund.status != RefundStatus.NEW:
            return RefundOrderEvents.FAILED, {'set__status': RefundStatus.FAILED}
        if refund.is_external:
            return RefundOrderEvents.OFFICE_REFUND, {'set__status': RefundStatus.PARTNER_REFUND_DONE}

        order = refund.order
        order.modify(set__travel_status=TravelOrderStatus.IN_PROGRESS)
        contract = ClientContracts.get_active_contract(order.partner)
        if contract is None:
            return RefundOrderEvents.NO_ACTIVE_CONTRACT, {'set__status': RefundStatus.FAILED}

        try:
            order_info = try_hard()(get_order_info)(order)
        except PartnerError:
            return RefundOrderEvents.FAILED, {'set__status': RefundStatus.FAILED}

        refund_managers = self.get_all_refund_managers(order_info)

        if not refund.has_sending_attempt:
            set_refund_has_sending_attempt(refund)
            for manager in refund_managers:
                manager.do_refund()
        else:
            # Упали на рефунде и не знаем посылали мы его или нет
            for manager in refund_managers:
                manager.update_refund_info()
                if manager.state == RefundManagerStates.NOT_FOUND:
                    manager.do_refund()

        def has_unknown_refunds():
            return any(manager.state == RefundManagerStates.UNKNOWN for manager in refund_managers)

        if has_unknown_refunds():
            update_refund_status(refund, RefundStatus.PARTNER_REFUND_UNKNOWN)

        # TODO: возможно сократить количество запросов в ИМ если спрашивать order_info 1 раз для всех возвратов
        retries_left = CHECK_ORDER_REFUND_PARAMS[order.partner]['MAX_RETRIES']
        delay_between_retries = CHECK_ORDER_REFUND_PARAMS[order.partner]['DELAY_BETWEEN_RETRIES'].total_seconds()
        while has_unknown_refunds() and retries_left > 0:
            retries_left -= 1
            sleep(delay_between_retries)
            for manager in refund_managers:
                manager.update_refund_info_if_unknown()

        refund_results = [manager.result for manager in refund_managers if manager.state == RefundManagerStates.DONE]
        refund_by_blank_id = {}
        for result in refund_results:
            refund_by_blank_id.update(result.refund_by_blank_id)
        refunded_blanks = set(refund_by_blank_id.keys())
        refunded_insurance_ids_from_blanks = [p.insurance.operation_id for p in order.passengers
                                              if p.tickets[0].blank_id in refunded_blanks
                                              and p.insurance
                                              and p.insurance.operation_id
                                              and not p.insurance.refund_uuid]
        refunded_insurance_ids = list(set(refund.insurance_ids + refunded_insurance_ids_from_blanks))
        self.update_order(refund_by_blank_id, refunded_insurance_ids)

        if all(manager.state == RefundManagerStates.DONE for manager in refund_managers):
            return RefundOrderEvents.DONE, {
                'set__status': RefundStatus.PARTNER_REFUND_DONE,
                'set__insurance_ids': refunded_insurance_ids,
            }
        elif refund_results:
            log.error('Внештатная ситуация. Часть билетов вернули, а часть не удалось вернуть, '
                      'нужно разбираться вручную. По билетам, которые вернули, суммы нужно смотреть в логе.')
            return RefundOrderEvents.FAILED, {
                'set__status': RefundStatus.FAILED,
                'set__is_partial_failed': True,
            }
        else:
            return RefundOrderEvents.FAILED, {'set__status': RefundStatus.FAILED}

    def get_all_refund_managers(self, order_info):
        refund = self.document
        refund_uuid = refund.uuid
        order = refund.order
        refund_blank_ids = refund.blank_ids
        refund_insurance_ids = refund.insurance_ids

        managers = []
        if refund_blank_ids:
            if self.need_to_return_tickets_by_one(order_info):
                managers.extend([BlankRefundManager(
                    order=order,
                    passenger=next(p for p in order_info.passengers if p.blank_id == blank_id),
                    blank_id=blank_id,
                    refund_uuid=refund_uuid,

                ) for blank_id in refund_blank_ids])
            else:
                passenger = next(p for p in order_info.passengers)
                managers.append(BlankRefundManager(order, passenger, blank_id=None, refund_uuid=refund_uuid))

        insurance_statuses = {i.operation_id: i.operation_status for i in order.iter_insurances()}
        managers.extend([InsuranceRefundManager(
            order=order,
            insurance_id=insurance_id,
            refund_uuid=refund_uuid,
        ) for insurance_id in refund_insurance_ids if insurance_statuses[insurance_id] != OperationStatus.FAILED])

        return managers

    def update_order(self, refund_by_blank_id, refunded_insurance_ids):
        refund = self.document
        update_spec = {}
        update_spec.update(build_order_update(refund.order, refund_by_blank_id))
        update_spec.update(build_order_insurance_update(refund.uuid, refund.order, refunded_insurance_ids))
        if update_spec:
            TrainOrder.objects.get(uid=refund.order_uid).update(**update_spec)

    def need_to_return_tickets_by_one(self, order_info):
        refund = self.document
        if len(refund.blank_ids) == 1:
            return True

        if {p.blank_id for p in order_info.passengers} != set(refund.blank_ids):
            return True

        # Для обратного выезда из Финляндии возврат доступен только отдельно по каждому билету,
        # возврат всего заказа полностью невозможен.
        from_country_id = Station.objects.values_list('country_id', flat=True).get(id=refund.order.station_from_id)
        if from_country_id == Country.FINLAND_ID:
            return True

        return False


def update_refund_status(refund, status):
    TrainRefund.objects.get(uuid=refund.uuid).update(set__status=status)


def set_refund_has_sending_attempt(refund):
    TrainRefund.objects.get(uuid=refund.uuid).update(set__has_sending_attempt=True)


def build_order_update(order, refund_by_blank_id):
    update_spec = {}
    for ticket, ticket_lookup_name in order.iter_ticket_to_lookup_name():
        if ticket.blank_id not in refund_by_blank_id:
            continue
        ticket_refund_result = refund_by_blank_id[ticket.blank_id]
        # TODO: здесь мы уже знаем id бланка возврата ticket_refund_result.refund_blank_id
        # Можно вписать его в заказ и использовать для отправки письма
        if ticket.payment.amount > 0:
            update_spec['set__{}__refund__amount'.format(ticket_lookup_name)] = ticket_refund_result.amount
        update_spec.update({
            'set__{}__refund__operation_id'.format(ticket_lookup_name): ticket_refund_result.refund_operation_id,
            # в случае успеха можем сами выставить статус
            'set__{}__rzhd_status'.format(ticket_lookup_name): RzhdStatus.REFUNDED,
        })
    return update_spec


def build_order_insurance_update(refund_uuid, order, refunded_insurance_ids):
    update_spec = {}
    for i, passenger in enumerate(order.passengers):
        if (passenger.insurance and passenger.insurance.operation_id in refunded_insurance_ids
                and passenger.insurance.trust_order_id):
            update_spec['set__passengers__{}__insurance__refund_uuid'.format(i)] = refund_uuid
    return update_spec
