# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import json
import operator
from decimal import Decimal
from itertools import chain
from logging import getLogger

import requests
import six
from django.conf import settings
from django.utils.encoding import force_text
from django.utils.http import urlencode

from common.apps.train.models import CoachType, Facility
from common.apps.train_order.enums import CoachType as CoachTypeEnum
from common.dynamic_settings.default import conf
from common.models.schedule import Company
from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from common.utils.date import UTC_TZ

from travel.rasp.train_api.train_partners.im.base import PROVIDER_P2
from travel.rasp.train_api.pb_provider import PROTOBUF_DATA_PROVIDER

define_setting('TRAIN_WIZARD_API_INDEXER_TIMEOUT', default=5)
define_setting('TRAIN_WIZARD_API_INDEXER_HOST', {
    Configuration.PRODUCTION: 'production.search-api.trains.internal.yandex.ru',
    Configuration.TESTING: 'testing.search-api.trains.internal.yandex.ru',
}, default=None)

define_setting('TRAIN_WIZARD_API_DIRECTION_TIMEOUT', default=5)
define_setting('TRAIN_WIZARD_API_DIRECTION_HOST', {
    Configuration.PRODUCTION: 'production.search-api.trains.internal.yandex.ru',
    Configuration.TESTING: 'testing.search-api.trains.internal.yandex.ru',
}, default=None)


def _dump_segments(segments):
    result = []
    for segment in segments:
        dump = {
            'arrival_dt': segment.arrival.isoformat(),
            'arrival_station_id': segment.station_to.id,
            'departure_dt': segment.departure.isoformat(),
            'departure_station_id': segment.station_from.id,
            'number': segment.original_number,
            'display_number': segment.number,
            'has_dynamic_pricing': segment.has_dynamic_pricing,
            'two_storey': segment.two_storey,
            'is_suburban': segment.is_suburban,
            'coach_owners': segment.coach_owners,
            'electronic_ticket': bool(segment.tariffs.get('electronic_ticket')),
            'first_country_code': segment.first_country_code or (segment.thread.first_country_code if segment.thread else None),
            'last_country_code': segment.last_country_code or (segment.thread.last_country_code if segment.thread else None),
            'places': [
                {
                    'coach_type': coach_type,
                    'count': tariff.seats,
                    'service_class': tariff.service_class,
                    'lower_count': tariff.lower_seats,
                    'upper_count': tariff.upper_seats,
                    'lower_side_count': tariff.lower_side_seats,
                    'upper_side_count': tariff.upper_side_seats,
                    'max_seats_in_the_same_car': tariff.max_seats_in_the_same_car,
                    'has_non_refundable_tariff': tariff.has_non_refundable_tariff,
                    'price': {
                        'currency': tariff.total_price.currency,
                        'value': force_text(tariff.total_price.value),
                    },
                    'price_details': {
                        'ticket_price': force_text(tariff.ticket_price.value),
                        'service_price': force_text(tariff.service_price.value),
                        'fee': force_text(tariff.fee.value),
                        'several_prices': tariff.several_prices,
                    },
                }
                for coach_type, tariff in segment.tariffs['classes'].items()
            ],
            'broken_classes': segment.tariffs.get('broken_classes'),
            'title_dict': json.loads(segment.title_common) if segment.title_common else None,
            'provider': segment.provider,
            'raw_train_name': segment.raw_train_name,
        }
        if segment.provider == PROVIDER_P2 and segment.thread:
            dump['number'] = segment.thread.number
        result.append(dump)
    return result


def _make_places_schema(coach_schema):
    group_numbers = {place_schema.number: place_schema.group_number for place_schema in coach_schema.places}
    if conf.TRAIN_BACKEND_USE_PROTOBUFS['facility']:
        facilities = {facility.Code: facility.Id for facility in PROTOBUF_DATA_PROVIDER.facility_repo.itervalues()}
    else:
        facilities = {facility.code: facility.id for facility in Facility.objects.all_cached()}

    places_facilities_ids = {}
    for place_flag, place_numbers in coach_schema.place_flags.items():
        facility_id = facilities.get(place_flag)
        if facility_id is not None:
            for place_number in place_numbers:
                places_facilities_ids.setdefault(place_number, set()).add(facility_id)

    places_schema = {}
    for place_number in six.viewkeys(group_numbers) | six.viewkeys(places_facilities_ids):
        places_schema[place_number] = place_schema = {}

        group_number = group_numbers.get(place_number)
        if group_number is not None:
            place_schema['group_number'] = group_number

        facilities_ids = places_facilities_ids.get(place_number)
        if facilities_ids is not None:
            place_schema['facilities_ids'] = sorted(facilities_ids)

    return places_schema


def _dump_place(place):
    price = place.adult_tariff
    return {
        'number': place.number,
        'price': {
            'value': float(price.value),
            'currency': price.currency
        },
    }


def _dump_tariff_errors(errors):
    return sorted(error.value for error in errors)


def _dump_coach(coach, schema_data):
    company_id = None
    try:
        company_id = Company.objects.get(express_code=coach.owner).id
    except Company.DoesNotExist:
        pass

    coach_type = CoachType.objects.get(code=coach.type)
    if conf.TRAIN_BACKEND_USE_PROTOBUFS['facility']:
        facility_repo = PROTOBUF_DATA_PROVIDER.facility_repo
        facilities_ids = []

        for code in coach.facilities:
            proto = facility_repo.get(code.encode('utf-8'))  # Workaround: TRAINS-5585
            if proto:
                facilities_ids.append(proto.Id)
    else:
        facilities_ids = Facility.objects.filter(code__in=coach.facilities).values_list('id', flat=True)

    return {
        'company_id': company_id,
        'errors': _dump_tariff_errors(coach.errors),
        'facilities': sorted(coach.facilities),  # deprecated
        'facilities_ids': sorted(facilities_ids),
        'number': coach.number,
        'owner': coach.owner,  # deprecated
        'places': [
            dict(schema_data.get(place.number, {}), **_dump_place(place))
            for place in sorted(coach.places, key=operator.attrgetter('number'))
        ],
        'type': coach.type,  # deprecated
        'type_id': coach_type.id,
    }


def _dump_details(train_details):
    place_schemas = {schema.id: _make_places_schema(schema) for schema in train_details.schemas}
    coaches = [
        _dump_coach(coach, place_schemas.get(coach.schema_id, {}))
        for coach in sorted(
            chain(train_details.coaches, train_details.broken_coaches),
            key=operator.attrgetter('number'),
        ) if CoachTypeEnum(coach.type) != CoachTypeEnum.UNKNOWN
    ]
    return {
        'coaches': coaches,
        'electronic_ticket': bool(train_details.electronic_ticket),
    }


class TrainWizardApiClient(object):
    def __init__(self, transport, logger):
        self._transport = transport
        self._logger = logger

    def _post(self, query, data):
        host = settings.TRAIN_WIZARD_API_INDEXER_HOST
        if host is None:
            self._logger.warn('Train wizard api host is not initialized')
            return

        self._logger.info('Start index: [%s]', query)
        try:
            response = self._transport.post(
                'https://{host}{query}'.format(host=host, query=query),
                headers={'Content-Type': 'application/json'},
                json=data,
                timeout=settings.TRAIN_WIZARD_API_INDEXER_TIMEOUT,
            )
            response.raise_for_status()
        except Exception:
            self._logger.warn('Can not index document to train wizard api: [%s]', query, exc_info=True)
        else:
            self._logger.info('Finish index: [%s]', query)

    def store_tariffs(self, tariffs):
        try:
            query = '/indexer/public-api/direction/?{}'.format(urlencode((
                ('departure_point_express_id', tariffs.query.departure_point_code),
                ('arrival_point_express_id', tariffs.query.arrival_point_code),
                ('departure_date', tariffs.query.departure_date),
            )))
            self._post(query, _dump_segments(tariffs.segments))
        except Exception:
            self._logger.exception('Could not index train direction tariffs')

    def store_details(self, details):
        try:
            query = '/indexer/public-api/train/?{}'.format(urlencode((
                ('departure_point_express_id', details.express_from),
                ('arrival_point_express_id', details.express_to),
                ('departure_dt', details.departure.astimezone(UTC_TZ).isoformat()),
                ('number', details.ticket_number)
            )))
            self._post(query, _dump_details(details))
        except Exception:
            self._logger.exception('Could not index train details of %s', details.ticket_number)

    def get_raw_segments_with_tariffs(self, departure_point_key, arrival_point_key, departure_date, safe=True):
        """
        Функция получения данных о сегментах и ценах поездов из источника колдунщика

        :param departure_point_key: ключ точки отправления, например 'c54', 's690034'
        :param arrival_point_key: ключ точки прибытия, например 'c54', 's690034'
        :param departure_date: дата отправления во временной зоне точки отправления
        :param safe: подавление ошибок
        :return: в случае успеха - сырой ответ ручки колдунщика, иначе - None
        """
        departure_date_str = departure_date.strftime('%Y-%m-%d')
        return self._get(
            path='/searcher/public-api/open_direction/',
            params={
                'departure_point_key': departure_point_key,
                'arrival_point_key': arrival_point_key,
                'departure_date': departure_date_str,
                'order_by': 'departure',
            },
            error_message='Could not get direction data {}-{} {}'.format(
                departure_point_key, arrival_point_key, departure_date_str
            ),
            safe=safe,
        )

    def _get(self, path, params=None, error_message='Ошибка получения данных', safe=True):
        try:
            host = settings.TRAIN_WIZARD_API_DIRECTION_HOST
            if host is None:
                raise Exception('Train wizard api host is not initialized')
            response = self._transport.get(
                url='https://{host}{path}'.format(host=host, path=path),
                params=params or {},
                headers={'Content-Type': 'application/json'},
                timeout=settings.TRAIN_WIZARD_API_DIRECTION_TIMEOUT,
                verify=settings.YANDEX_INTERNAL_ROOT_CA,
            )
            response.raise_for_status()
            if response.status_code == 204:
                self._logger.info(error_message + '. 204 - Empty result.')
                return None
            return response.json(parse_float=Decimal)
        except Exception:
            if safe:
                self._logger.exception(error_message)
                return None
            raise

    def get_raw_prices_by_direction(self, directions, departure_date_from, departure_date_to, tld):
        """
        Функция получения данных о ценах поездов из источника колдунщика

        :param directions: массив пар ключей отправления и назначения, например [('c54', 's69003'), ('c2', 'c5')]
        :param departure_date_from: дата отправления во временной зоне точки отправления - нижняя граница
        :param departure_date_to: дата отправления во временной зоне точки отправления - верхняя граница
        :return: в случае успеха - сырой ответ ручки колдунщика, иначе - None
        """
        departure_date_from_str = departure_date_from.strftime('%Y-%m-%dT%H:%M')
        departure_date_to_str = departure_date_to.strftime('%Y-%m-%dT%H:%M')
        return self._get(
            path='/searcher/public-api/prices_by_directions/',
            params={
                'departure_points': [pair[0] for pair in directions],
                'arrival_points': [pair[1] for pair in directions],
                'departure_date_from': departure_date_from_str,
                'departure_date_to': departure_date_to_str,
                'order_by': 'departure',
                'tld': tld or 'ru',
            },
            error_message='Could not get prices data {}-{} {}'.format(
                departure_date_from_str, departure_date_to_str, directions
            ),
            safe=True,
        )


train_wizard_api_client = TrainWizardApiClient(
    logger=getLogger(__name__),
    transport=requests
)
