package main

import (
	"context"
	"fmt"
	"os"
	"sync"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/library/go/tracing"
	"a.yandex-team.ru/travel/library/go/tvm"

	"a.yandex-team.ru/travel/rasp/train_bandit_api/internal/api/handler"
	"a.yandex-team.ru/travel/rasp/train_bandit_api/internal/api/server"
	"a.yandex-team.ru/travel/rasp/train_bandit_api/internal/bandit/manager"
	"a.yandex-team.ru/travel/rasp/train_bandit_api/internal/logging"
	"a.yandex-team.ru/travel/rasp/train_bandit_api/internal/pricing"
)

type Config struct {
	HTTPConfig    *server.HTTPConfig     `yaml:"http"`
	GRPCConfig    *server.GRPCConfig     `yaml:"grpc"`
	LogConfig     *logging.Config        `yaml:"log"`
	PricingConfig *pricing.PricingConfig `yaml:"pricing"`
	TvmConfig     *tvm.TvmHelperConfig   `yaml:"tvm"`
	ManagerConfig *manager.Config        `yaml:"manager"`
	Metrics       *metricserver.MetricsConfig
}

var cfg = Config{
	HTTPConfig:    &server.DefaultHTTPConfig,
	GRPCConfig:    &server.DefaultGRPCConfig,
	LogConfig:     &logging.DefaultConfig,
	PricingConfig: &pricing.DefaultPricingConfig,
	ManagerConfig: &manager.DefaultConfig,
	Metrics:       &metricserver.DefaultMetricsConfig,
}

func main() {
	maxprocs.AdjustAuto()
	loader := configuration.NewDefaultConfitaLoader()
	ctx, cancel := context.WithCancel(context.Background())
	err := loader.Load(ctx, &cfg)
	if err != nil {
		fmt.Println("Failed to load config:", err)
		os.Exit(1)
	}
	logger, err := logging.New(cfg.LogConfig)
	if err != nil {
		fmt.Println("Failed to create logger, err:", err)
		os.Exit(1)
	}
	err = logging.EnableLogrotate()
	if err != nil {
		logger.Fatal("Failed to enable log rotation, err:", log.Error(err))
	}
	tracerCloser := tracing.InitializeDefaultTracer("train_bandit_api.api")

	metricsRegistry := metrics.NewRegistryWithDeployTags()
	appMetricsRegistry := metricsRegistry.WithPrefix("app")
	appMetrics := metrics.NewAppMetrics(appMetricsRegistry)
	metrics.SetGlobalAppMetrics(appMetrics)

	defer func() {
		err = tracerCloser.Close()
		if err != nil {
			logger.Error("Tracer close error:", log.Error(err))
		}
		err = logger.L.Sync()
		if err != nil {
			fmt.Println("Failed to close logger:", err)
			return
		}
		cancel()
	}()

	tvmHelper := tvm.NewDeployTvmHelper(logger, cfg.TvmConfig)

	waitGroup := sync.WaitGroup{}
	waitGroup.Add(2)
	go func() {
		defer waitGroup.Done()

		err = server.RunHTTPServer(cfg.HTTPConfig, logger, metricsRegistry.WithPrefix("http"))
		if err != nil {
			logger.Fatal("Cannot run HTTPServer:", log.Error(err))
		}
	}()

	bm, err := manager.NewManager(cfg.ManagerConfig, logger)
	if err != nil {
		logger.Fatal("Failed to create bandit manager, err:", log.Error(err))
	}
	pricer, err := pricing.New(cfg.PricingConfig)
	if err != nil {
		logger.Fatal("Failed to create pricer, err:", log.Error(err))
	}

	grpcHandler := handler.GRPCHandler{
		ChargerGetter: bm,
		Logger:        logger,
		Pricer:        pricer,
		Metrics:       appMetrics,
	}

	go func() {
		defer waitGroup.Done()

		err = server.RunGRPCServer(cfg.GRPCConfig, &grpcHandler, logger, metricsRegistry.WithPrefix("grpc"), &tvmHelper)
		if err != nil {
			logger.Fatal("Cannot run GRPCServer", log.Error(err))
		}
	}()

	go func() {
		defer waitGroup.Done()

		err = metricserver.RunMetricsHTTPServer(context.Background(), *cfg.Metrics, logger, metricsRegistry)
		if err != nil {
			logger.Fatal("Error while starting metrics server", log.Error(err))
		}
	}()

	logger.Info("Started")
	waitGroup.Wait()
}
