package main

import (
	"context"
	"flag"
	"fmt"
	"os"
	"sync"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/library/go/tracing"
	"a.yandex-team.ru/travel/library/go/tvm"

	"a.yandex-team.ru/travel/rasp/train_offer_storage/internal/api/handler"
	"a.yandex-team.ru/travel/rasp/train_offer_storage/internal/api/server"
	"a.yandex-team.ru/travel/rasp/train_offer_storage/internal/logging"
	"a.yandex-team.ru/travel/rasp/train_offer_storage/internal/redir"
	"a.yandex-team.ru/travel/rasp/train_offer_storage/internal/storage"
)

type Config struct {
	HTTPConfig    *server.HTTPConfig          `yaml:"http"`
	GRPCConfig    *server.GRPCConfig          `yaml:"grpc"`
	LogConfig     *logging.Config             `yaml:"log"`
	StorageConfig *storage.Config             `yaml:"storage"`
	TvmConfig     *tvm.TvmHelperConfig        `yaml:"tvm"`
	RedirConfig   *redir.Config               `yaml:"redir"`
	Metrics       *metricserver.MetricsConfig `yaml:"metrics"`
}

var cfg = Config{
	HTTPConfig:    &server.DefaultHTTPConfig,
	GRPCConfig:    &server.DefaultGRPCConfig,
	LogConfig:     &logging.DefaultConfig,
	StorageConfig: &storage.DefaultConfig,
	RedirConfig:   &redir.DefaultConfig,
	Metrics:       &metricserver.DefaultMetricsConfig,
}

func main() {
	maxprocs.AdjustAuto()
	consoleFlag := flag.Bool("console", false, "enter console mode")
	loader := configuration.NewDefaultConfitaLoader()
	ctx, cancel := context.WithCancel(context.Background())
	err := loader.Load(ctx, &cfg)
	if err != nil {
		fmt.Println("failed to load config:", err)
		os.Exit(1)
	}
	logger, err := logging.New(cfg.LogConfig)
	if err != nil {
		fmt.Println("failed to create logger, err:", err)
		os.Exit(1)
	}
	tracerCloser := tracing.InitializeDefaultTracer("train_offer_storage.api")
	storager, err := storage.NewStorage(ctx, cfg.StorageConfig, logger)
	if err != nil {
		logger.Fatal("storage init error:", log.Error(err))
	}
	redirService := redir.NewService(cfg.RedirConfig, logger)

	defer func() {
		err = storager.Close(ctx)
		if err != nil {
			logger.Error("storage close error:", log.Error(err))
		}
		err = tracerCloser.Close()
		if err != nil {
			logger.Error("tracer close error:", log.Error(err))
		}
		err = logger.L.Sync()
		if err != nil {
			fmt.Println("failed to close logger:", err)
			return
		}
		cancel()
	}()

	if *consoleFlag {
		console(ctx, logger, storager)
		return
	}
	tvmHelper := tvm.NewDeployTvmHelper(logger, cfg.TvmConfig)
	wg := sync.WaitGroup{}
	wg.Add(2)

	httpHandler := handler.HTTPHandler{
		Logger:   logger,
		Storager: storager,
		Redir:    redirService,
	}

	metricsRegistry := metrics.NewRegistryWithDeployTags()
	metrics.RunPerfMetricsUpdater(metricsRegistry, cfg.Metrics.PerfMetricsRefreshInterval)

	go func() {
		defer wg.Done()

		err = server.RunHTTPServer(cfg.HTTPConfig, &httpHandler, logger, metricsRegistry.WithPrefix("http"), &tvmHelper)
		if err != nil {
			logger.Fatal("Cannot run HttpServer:", log.Error(err))
		}
	}()

	grpcHandler := handler.GRPCHandler{
		Logger:   logger,
		Storager: storager,
	}

	go func() {
		defer wg.Done()

		err = server.RunGRPCServer(cfg.GRPCConfig, &grpcHandler, logger, metricsRegistry.WithPrefix("grpc"), &tvmHelper)
		if err != nil {
			logger.Fatal("Cannot run GrpcServer", log.Error(err))
		}
	}()

	go func() {
		defer wg.Done()
		err = metricserver.RunMetricsHTTPServer(context.Background(), *cfg.Metrics, logger, metricsRegistry)
		if err != nil {
			logger.Fatal("Error while starting metrics server", log.Error(err))
		}
	}()

	logger.Info("Started")
	wg.Wait()
}
