package handler

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/proto/trains"
	"a.yandex-team.ru/travel/rasp/train_offer_storage/internal/api/models"
	"a.yandex-team.ru/travel/rasp/train_offer_storage/proto"
	"github.com/gofrs/uuid"
	"github.com/golang/protobuf/proto"
	"github.com/opentracing/opentracing-go"
	"golang.org/x/net/context"
)

type GRPCHandler struct {
	Logger   log.Logger
	Storager models.Storager
}

func (h *GRPCHandler) GetOffer(ctx context.Context, request *train_offer_proto.TGetOfferRequest) (
	response *train_offer_proto.TGetOfferResponse, err error) {
	span, ctx := opentracing.StartSpanFromContext(ctx, "api.handler.GetOffer")
	defer span.Finish()

	token, err := uuid.FromString(request.OfferId)
	if err != nil {
		h.Logger.Error("Cannot parse offerID", log.Error(err))
		return nil, err
	}

	data, err := h.Storager.Get(ctx, token)
	if err != nil {
		h.Logger.Error("Cannot get data from storage", log.Error(err))
		return nil, err
	}

	offer := new(trains.TTrainServiceOffer)
	err = proto.Unmarshal(data, offer)
	if err != nil {
		h.Logger.Error("Cannot unmarshal data", log.Error(err))
		return nil, err
	}

	return &train_offer_proto.TGetOfferResponse{Offer: offer}, nil
}

func (h *GRPCHandler) GetOffers(ctx context.Context, request *train_offer_proto.TGetOffersRequest) (
	response *train_offer_proto.TGetOffersResponse, err error) {
	span, ctx := opentracing.StartSpanFromContext(ctx, "api.handler.GetOffers")
	defer span.Finish()

	var tokens []uuid.UUID
	var offers []*trains.TTrainServiceOffer

	for _, offerID := range request.GetOfferIds() {
		token, err := uuid.FromString(offerID)
		if err != nil {
			h.Logger.Error("Cannot parse offerID", log.Error(err))
			return nil, err
		}
		tokens = append(tokens, token)
	}

	if len(tokens) > 0 {
		datas, err := h.Storager.GetMany(ctx, tokens)
		if err != nil {
			h.Logger.Error("Cannot get data from storage", log.Error(err))
			return nil, err
		}

		for _, data := range datas {
			offer := new(trains.TTrainServiceOffer)
			err = proto.Unmarshal(data, offer)
			if err != nil {
				h.Logger.Error("Cannot unmarshal data", log.Error(err))
				return nil, err
			}
			offers = append(offers, offer)
		}
	}

	return &train_offer_proto.TGetOffersResponse{Offers: offers}, nil
}
