package handler

import (
	"encoding/json"
	"errors"
	"fmt"
	"net/http"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/httputil"
	"github.com/gofrs/uuid"
	"github.com/golang/protobuf/proto"
	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/travel/rasp/train_offer_storage/internal/api/models"
	"a.yandex-team.ru/travel/rasp/train_offer_storage/internal/redir"
)

type HTTPHandler struct {
	Logger   log.Logger
	Storager models.Storager
	Redir    *redir.Service
}

func (h *HTTPHandler) Store(w http.ResponseWriter, r *http.Request) {
	span, ctx := opentracing.StartSpanFromContext(r.Context(), "api.handler.Store")
	defer span.Finish()
	decoder := json.NewDecoder(r.Body)
	var request models.StoreOfferRequest
	err := decoder.Decode(&request)
	if err != nil {
		h.Logger.Error("parse store request error", log.Error(err))
		httputil.HandleError(err, http.StatusBadRequest, w)
		return
	}
	err = request.Validate()
	if err != nil {
		h.Logger.Error("store request validation error", log.Error(err))
		httputil.HandleError(err, http.StatusBadRequest, w)
		return
	}
	if request.BanditType == "" {
		expStr := r.Header.Get(httputil.UaasExperimentsHeader)
		experiments, err := httputil.ParseExperiments(expStr)
		if err != nil {
			h.Logger.Errorf("error parse experiments='%s': %s", expStr, err.Error())
		} else {
			request.BanditType = experiments.TrainsBanditType
		}
	}
	uid, err := uuid.NewV4()
	if err != nil {
		panic(fmt.Errorf("uuid generation error: %w", err))
	}
	offer, err := request.ToProtoOffer(uid)
	if err != nil {
		h.Logger.Error("offer data mapping error", log.Error(err))
		httputil.HandleError(err, http.StatusBadRequest, w)
		return
	}
	data, err := proto.Marshal(offer)
	if err != nil {
		panic(fmt.Errorf("offer marshal error: %w", err))
	}
	err = h.Storager.Save(ctx, data, uid)
	if err != nil {
		h.Logger.Error("save offer error", log.Error(err))
		err = errors.New("save offer error")
		httputil.HandleError(err, http.StatusInternalServerError, w)
		return
	}
	lh, err := h.Redir.StoreLabelParams(ctx, request.ToLabelParams())
	if err != nil {
		h.Logger.Error("save label params error", log.Error(err))
		err = errors.New("save label params error")
		httputil.HandleError(err, http.StatusInternalServerError, w)
		return
	}
	response := models.StoreOfferResponse{
		OfferID:   uid.String(),
		LabelHash: lh,
	}
	w.WriteHeader(http.StatusOK)
	err = json.NewEncoder(w).Encode(response)
	if err != nil {
		panic(fmt.Errorf("encode response error: %w", err))
	}
}
