package models

import (
	"fmt"
	"time"

	"a.yandex-team.ru/travel/proto/trains"
	"github.com/gofrs/uuid"
	"github.com/golang/protobuf/ptypes"
	"github.com/golang/protobuf/ptypes/timestamp"

	"a.yandex-team.ru/travel/rasp/train_bandit_api/pkg/utils"
	bpr "a.yandex-team.ru/travel/rasp/train_bandit_api/proto"
)

var (
	RoutePolicyMap = map[string]trains.ERoutePolicy{
		"internal":      trains.ERoutePolicy_RP_INTERNAL,
		"international": trains.ERoutePolicy_RP_INTERNATIONAL,
		"finland":       trains.ERoutePolicy_RP_FINLAND,
	}
	CabinPlaceDemandsMap = map[string]trains.ECabinPlaceDemands{
		"compartment": trains.ECabinPlaceDemands_PD_IN_ONE_CABIN,
		"section":     trains.ECabinPlaceDemands_PD_IN_ONE_CABIN,
		"nearest":     trains.ECabinPlaceDemands_PD_NO_SIDE_PLACES,
		"notSide":     trains.ECabinPlaceDemands_PD_IN_ONE_COMPARTMENT,
	}
	CarStoreyMap = map[uint32]trains.ECarStorey{
		0: trains.ECarStorey_CS_UNKNOWN,
		1: trains.ECarStorey_CS_FIRST,
		2: trains.ECarStorey_CS_SECOND,
	}
	PassengerCategoryMap = map[string]trains.EPassengerCategory{
		"adults":   trains.EPassengerCategory_PC_ADULT,
		"children": trains.EPassengerCategory_PC_CHILD,
		"babies":   trains.EPassengerCategory_PC_BABY,
	}
	DocumentTypeMap = map[string]trains.EDocumentType{
		"ПН": trains.EDocumentType_DT_RUSSIAN_PASSPORT,
		"СР": trains.EDocumentType_DT_BIRTH_CERTIFICATE,
		"ЗП": trains.EDocumentType_DT_RUSSIAN_INTERNATIONAL_PASSPORT,
		"ЗЗ": trains.EDocumentType_DT_FOREIGN_DOCUMENT,
		"ПМ": trains.EDocumentType_DT_SAILOR_PASSPORT,
		"ВБ": trains.EDocumentType_DT_MILITARY_CARD,
	}
	SexMap = map[string]trains.ESex{
		"M": trains.ESex_S_MALE,
		"F": trains.ESex_S_FEMALE,
	}
	LoyaltyCardTypeMap = map[string]trains.ELoyaltyCardType{
		"RzhdB": trains.ELoyaltyCardType_LCT_RZHD_BONUS,
		"RzhdU": trains.ELoyaltyCardType_LCT_UNIVERSAL_RZHD_CARD,
	}
	CarTypeMap = map[string]trains.ECarType{
		"platzkarte":  trains.ECarType_CT_RESERVED_SEAT,
		"compartment": trains.ECarType_CT_COMPARTMENT,
		"suite":       trains.ECarType_CT_LUXURY,
		"common":      trains.ECarType_CT_SHARED,
		"sitting":     trains.ECarType_CT_SEDENTARY,
		"soft":        trains.ECarType_CT_SOFT,
	}
	CabinGenderKindMap = map[string]trains.ECabinGenderKind{
		"male":   trains.ECabinGenderKind_GK_MALE,
		"female": trains.ECabinGenderKind_GK_FEMALE,
		"mixed":  trains.ECabinGenderKind_GK_MIXED,
	}
	AdditionalPlaceRequirementsMap = map[string]trains.EAdditionalPlaceRequirements{
		"NoValue":                         trains.EAdditionalPlaceRequirements_PR_UNKNOWN,
		"MotherAndBabyPlaces":             trains.EAdditionalPlaceRequirements_PR_MOTHER_AND_BABY_PLACES,
		"WithBabyPlaces":                  trains.EAdditionalPlaceRequirements_PR_WITH_BABY_PLACES,
		"WithPetsPlaces":                  trains.EAdditionalPlaceRequirements_PR_WITH_PETS_PLACES,
		"Usual":                           trains.EAdditionalPlaceRequirements_PR_USUAL,
		"UsualNearTheTable":               trains.EAdditionalPlaceRequirements_PR_USUAL_NEAR_THE_TABLE,
		"AnyNearTheTable":                 trains.EAdditionalPlaceRequirements_PR_ANY_NEAR_THE_TABLE,
		"AnyNotNearTheTable":              trains.EAdditionalPlaceRequirements_PR_ANY_NOT_NEAR_THE_TABLE,
		"NearThePlayground":               trains.EAdditionalPlaceRequirements_PR_NEAR_THE_PLAYGROUND,
		"NearThePlaygroundAndNotTheTable": trains.EAdditionalPlaceRequirements_PR_NEAR_THE_PLAYGROUND_AND_NOT_THE_TABLE,
		"NearThePlaygroundAndTheTable":    trains.EAdditionalPlaceRequirements_PR_NEAR_THE_PLAYGROUND_AND_THE_TABLE,
		"NearThePlacesWithPets":           trains.EAdditionalPlaceRequirements_PR_NEAR_THE_PLACES_WITH_PETS,
		"FoldablePlace":                   trains.EAdditionalPlaceRequirements_PR_FOLDABLE_PLACE,
		"Forward":                         trains.EAdditionalPlaceRequirements_PR_FORWARD,
		"Backward":                        trains.EAdditionalPlaceRequirements_PR_BACKWARD,
		"NearWindow":                      trains.EAdditionalPlaceRequirements_PR_NEAR_WINDOW,
		"UnfoldablePlace":                 trains.EAdditionalPlaceRequirements_PR_UNFOLDABLE_PLACE,
		"NearTheTableAndBackward":         trains.EAdditionalPlaceRequirements_PR_NEAR_THE_TABLE_AND_BACKWARD,
		"NearTheTableAndForward":          trains.EAdditionalPlaceRequirements_PR_NEAR_THE_TABLE_AND_FORWARD,
		"WithoutTableAndBackward":         trains.EAdditionalPlaceRequirements_PR_WITHOUT_TABLE_AND_BACKWARD,
		"WithoutTableAndForward":          trains.EAdditionalPlaceRequirements_PR_WITHOUT_TABLE_AND_FORWARD,
		"WithoutWindowAndBackward":        trains.EAdditionalPlaceRequirements_PR_WITHOUT_WINDOW_AND_BACKWARD,
		"WithoutWindowAndForward":         trains.EAdditionalPlaceRequirements_PR_WITHOUT_WINDOW_AND_FORWARD,
		"SingleAndForward":                trains.EAdditionalPlaceRequirements_PR_SINGLE_AND_FORWARD,
		"NearRestroom":                    trains.EAdditionalPlaceRequirements_PR_NEAR_RESTROOM,
		"NearRestroomAndBackward":         trains.EAdditionalPlaceRequirements_PR_NEAR_RESTROOM_AND_BACKWARD,
		"NearRestroomAndForward":          trains.EAdditionalPlaceRequirements_PR_NEAR_RESTROOM_AND_FORWARD,
		"NoTableAndNoWindow":              trains.EAdditionalPlaceRequirements_PR_NO_TABLE_AND_NO_WINDOW,
	}
	DirectionMap = map[string]trains.EDirection{
		"FORWARD":  trains.EDirection_D_FORWARD,
		"BACKWARD": trains.EDirection_D_BACKWARD,
	}
)

func (req *StoreOfferRequest) ToProtoOffer(uid uuid.UUID) (offer *trains.TTrainServiceOffer, err error) {
	trainInfo := trains.TTrainInfo{
		TrainNumber:          req.TrainNumber,
		TrainTicketNumber:    req.TrainTicketNumber,
		TrainTitle:           req.TrainTitle,
		BrandTitle:           req.BrandTitle,
		IsCppk:               req.IsCppk,
		ImInitialStationName: req.ImInitialStationName,
		ImFinalStationName:   req.ImFinalStationName,
		Provider:             req.Provider,
	}
	var ok bool
	if trainInfo.RoutePolicy, ok = RoutePolicyMap[req.RoutePolicy]; !ok {
		return nil, fmt.Errorf("unknown routePolicy: %v", req.RoutePolicy)
	}
	var placeRequirements *trains.TPlaceRequirements = nil
	if req.Requirements != nil {
		var createOrderPlaceCount *trains.TPlaceCount = nil
		if req.Requirements.Count != nil {
			createOrderPlaceCount = &trains.TPlaceCount{
				Bottom:      req.Requirements.Count.Bottom,
				Upper:       req.Requirements.Count.Upper,
				NearWindow:  req.Requirements.Count.NearWindow,
				NearPassage: req.Requirements.Count.NearPassage,
			}
		}
		placeRequirements = &trains.TPlaceRequirements{
			CreateOrderPlaceCount: createOrderPlaceCount,
		}
		if req.Requirements.Arrangement != "" {
			if placeRequirements.CabinPlaceDemands, ok = CabinPlaceDemandsMap[req.Requirements.Arrangement]; !ok {
				return nil, fmt.Errorf("unknown arrangement: %v", req.Requirements.Arrangement)
			}
		}
		if placeRequirements.CarStorey, ok = CarStoreyMap[req.Requirements.Storey]; !ok {
			return nil, fmt.Errorf("unknown storey: %v", req.Requirements.Storey)
		}
	}
	priceDetails := trains.TPriceDetails{
		Price: utils.ToProtoPriceRUB(req.Amount),
	}
	passengers := make([]*trains.TPassenger, len(req.Passengers))
	for i := 0; i < len(req.Passengers); i++ {
		requestPassenger := req.Passengers[i]
		if passengers[requestPassenger.Index] != nil {
			return nil, fmt.Errorf("many passengers with same index %v", requestPassenger.Index)
		}
		passenger := &trains.TPassenger{
			CitizenshipGeoId:      requestPassenger.CitizenshipGeoID,
			TariffCode:            requestPassenger.Tariff,
			PassengerId:           requestPassenger.Index,
			IsNonRefundableTariff: requestPassenger.IsNonRefundableTariff,
			Places:                requestPassenger.Places,
		}
		if passenger.PassengerCategory, ok = PassengerCategoryMap[requestPassenger.AgeGroup]; !ok {
			return nil, fmt.Errorf("unknown AgeGroup: %v", requestPassenger.AgeGroup)
		}
		if passenger.DocumentType, ok = DocumentTypeMap[requestPassenger.DocType]; !ok {
			return nil, fmt.Errorf("unknown DocType: %v", requestPassenger.DocType)
		}
		if passenger.Sex, ok = SexMap[requestPassenger.Sex]; !ok {
			return nil, fmt.Errorf("unknown Sex: %v", requestPassenger.Sex)
		}
		if requestPassenger.LoyaltyCards != nil {
			lcs := make([]*trains.TLoyaltyCard, len(requestPassenger.LoyaltyCards))
			for j, card := range requestPassenger.LoyaltyCards {
				lcs[j] = &trains.TLoyaltyCard{
					Number: card.Number,
				}
				if lcs[j].Type, ok = LoyaltyCardTypeMap[card.CardType]; !ok {
					return nil, fmt.Errorf("unknown CardType: %v", card.CardType)
				}
			}
			passenger.LoyaltyCards = lcs
		}
		passengers[requestPassenger.Index] = passenger
	}
	departureTime, err := time.Parse(time.RFC3339, req.Departure)
	if err != nil {
		return nil, fmt.Errorf("invalid departure format: %w", err)
	}
	departureTimestamp, err := ptypes.TimestampProto(departureTime)
	if err != nil {
		return nil, fmt.Errorf("invalid departure format: %w", err)
	}
	arrivalTime, err := time.Parse(time.RFC3339, req.Arrival)
	if err != nil {
		return nil, fmt.Errorf("invalid arrival format: %w", err)
	}
	arrivalTimestamp, err := ptypes.TimestampProto(arrivalTime)
	if err != nil {
		return nil, fmt.Errorf("invalid arrival format: %w", err)
	}
	offer = &trains.TTrainServiceOffer{
		OfferId:                   uid.String(),
		Partner:                   req.Partner,
		TrainInfo:                 &trainInfo,
		CarNumber:                 req.CarNumber,
		CompanyTitle:              req.CompanyTitle,
		PlaceRequirements:         placeRequirements,
		Departure:                 departureTimestamp,
		Arrival:                   arrivalTimestamp,
		StationToId:               req.StationToID,
		StationFromId:             req.StationFromID,
		ServiceClass:              req.ServiceClass,
		SchemeId:                  req.SchemeID,
		InternationalServiceClass: req.InternationalServiceClass,
		Bedding:                   req.Bedding,
		ElectronicRegistration:    req.ElectronicRegistration,
		GiveChildWithoutPlace:     req.GiveChildWithoutPlace,
		Passengers:                passengers,
		PriceDetails:              &priceDetails,
		Places:                    req.Places,
		BanditType:                req.BanditType,
	}
	if req.CanChooseBedding != nil && !*req.CanChooseBedding {
		offer.BeddingChoose = trains.EBedding_B_ANY
	} else if req.Bedding {
		offer.BeddingChoose = trains.EBedding_B_WITH
	} else {
		offer.BeddingChoose = trains.EBedding_B_WITHOUT
	}
	if req.AdditionalPlaceRequirements != "" {
		if offer.AdditionalPlaceRequirements, ok = AdditionalPlaceRequirementsMap[req.AdditionalPlaceRequirements]; !ok {
			return nil, fmt.Errorf("unknown AdditionalPlaceRequirements: %v", req.CarType)
		}
	}
	if offer.CarType, ok = CarTypeMap[req.CarType]; !ok {
		return nil, fmt.Errorf("unknown CarType: %v", req.CarType)
	}
	if req.CabinGenderKind != "" {
		if offer.CabinGenderKind, ok = CabinGenderKindMap[req.CabinGenderKind]; !ok {
			return nil, fmt.Errorf("unknown CabinGenderKind: %v", req.CabinGenderKind)
		}
	}
	context := req.ToBanditContext(departureTimestamp, arrivalTimestamp)
	contextString, err := utils.MarshalToStr(context)
	if err != nil {
		return nil, fmt.Errorf("can not get string context: %w", err)
	}
	offer.BanditContext = contextString
	offer.FeeCalculationToken = req.FeeCalculationToken
	if req.Direction != "" {
		if offer.Direction, ok = DirectionMap[req.Direction]; !ok {
			return nil, fmt.Errorf("unknown Direction: %v", req.Direction)
		}
	}
	offer.SegmentIndex = req.SegmentIndex
	return offer, nil
}

func (req *StoreOfferRequest) ToBanditContext(departure, arrival *timestamp.Timestamp) *bpr.TContext {
	context := &bpr.TContext{
		ICookie:   req.LabelParams.ICookie,
		PointFrom: fmt.Sprintf("s%v", req.StationFromID),
		PointTo:   fmt.Sprintf("s%v", req.StationToID),
		Departure: departure,
		Arrival:   arrival,
		TrainType: req.RawTrainName,
		CarType:   req.CarType,
	}
	return context
}

func (req *StoreOfferRequest) ToLabelParams() *trains.TLabelParams {
	return &trains.TLabelParams{
		SerpReqId:        req.LabelParams.SerpReqID,
		UtmSource:        req.LabelParams.UtmSource,
		UtmMedium:        req.LabelParams.UtmMedium,
		UtmCampaign:      req.LabelParams.UtmCampaign,
		UtmTerm:          req.LabelParams.UtmTerm,
		UtmContent:       req.LabelParams.UtmContent,
		From:             req.LabelParams.From,
		Gclid:            req.LabelParams.GClID,
		ICookie:          req.LabelParams.ICookie,
		SerpUuid:         req.LabelParams.SerpUUID,
		TestBuckets:      req.LabelParams.TestBuckets,
		Partner:          req.LabelParams.Partner,
		SubPartner:       req.LabelParams.SubPartner,
		PartnerUid:       req.LabelParams.PartnerUID,
		Device:           req.LabelParams.Device,
		Terminal:         req.LabelParams.Terminal,
		IsTransfer:       req.LabelParams.IsTransfer,
		Ip:               req.LabelParams.IP,
		RegionId:         req.LabelParams.RegionID,
		Uid:              req.LabelParams.UID,
		YandexUid:        req.LabelParams.YandexUID,
		WizardReqId:      req.LabelParams.WizardReqID,
		SerpTestId:       req.LabelParams.SerpTestID,
		YtpReferer:       req.LabelParams.YtpReferer,
		Yclid:            req.LabelParams.Yclid,
		Fbclid:           req.LabelParams.Fbclid,
		MetrikaClientId:  req.LabelParams.MetrikaClientID,
		Clid:             req.LabelParams.Clid,
		AdmitadUid:       req.LabelParams.AdmitadUID,
		TravelpayoutsUid: req.LabelParams.TravelpayoutsUID,
		Vid:              req.LabelParams.Vid,
		AffiliateClid:    req.LabelParams.AffiliateClid,
		AffiliateVid:     req.LabelParams.AffliliateVid,
	}
}
