package server

import (
	"net"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/metrics"
	grpcmetrics "a.yandex-team.ru/travel/library/go/grpcutil/metrics"
	"a.yandex-team.ru/travel/library/go/tvm"
	middleware "github.com/grpc-ecosystem/go-grpc-middleware"
	grpcLog "github.com/grpc-ecosystem/go-grpc-middleware/logging/zap"
	recovery "github.com/grpc-ecosystem/go-grpc-middleware/recovery"
	grpc_opentracing "github.com/grpc-ecosystem/go-grpc-middleware/tracing/opentracing"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	ghealth "google.golang.org/grpc/health"
	"google.golang.org/grpc/health/grpc_health_v1"
	"google.golang.org/grpc/reflection"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/travel/rasp/train_offer_storage/internal/api/handler"
	"a.yandex-team.ru/travel/rasp/train_offer_storage/proto"
)

type GRPCConfig struct {
	Address         string
	AllowReflection bool
	HealthServer    bool
}

var DefaultGRPCConfig = GRPCConfig{
	Address:         ":9111",
	AllowReflection: true,
	HealthServer:    true,
}

const (
	serviceName = "train_offer_storage"
)

func RunGRPCServer(config *GRPCConfig, handler *handler.GRPCHandler, logger *zap.Logger, registry metrics.Registry,
	tvmHelper *tvm.TvmHelper) error {
	listener, err := net.Listen("tcp", config.Address)
	if err != nil {
		logger.Error("Failed to listen", log.Error(err))
		return err
	}
	recoveryOptions := []recovery.Option{
		recovery.WithRecoveryHandler(func(p interface{}) error {
			err := status.Errorf(codes.Internal, "Panic in grpc: %v", p)
			logger.Error("Recovery in grpc", log.Error(err))
			return err
		}),
	}
	interceptors := []grpc.UnaryServerInterceptor{
		grpc_opentracing.UnaryServerInterceptor(),
		grpcLog.UnaryServerInterceptor(logger.L),
		recovery.UnaryServerInterceptor(recoveryOptions...),
		grpcmetrics.NewGrpcMetricsInterceptor(registry),
	}
	tvmInterceptor := tvmHelper.CheckServiceTicketGRPCInterceptor()
	if tvmInterceptor != nil {
		interceptors = append(interceptors, tvmInterceptor)
	}
	server := grpc.NewServer(
		middleware.WithUnaryServerChain(interceptors...),
	)

	if config.AllowReflection {
		reflection.Register(server)
	}
	if config.HealthServer {
		healthServer := ghealth.NewServer()
		go func() {
			for range time.Tick(2 * time.Second) {
				healthServer.SetServingStatus(serviceName, grpc_health_v1.HealthCheckResponse_SERVING)
			}
		}()
		grpc_health_v1.RegisterHealthServer(server, healthServer)
	}

	train_offer_proto.RegisterOfferStorageApiServiceV1Server(server, handler)

	err = server.Serve(listener)
	if err != nil {
		logger.Error("Failed to serve", log.Error(err))
		return err
	}

	return nil
}
