package server

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/httputil/middleware/httpmetrics"
	"a.yandex-team.ru/travel/library/go/tracing"
	"a.yandex-team.ru/travel/library/go/tvm"
	"a.yandex-team.ru/travel/rasp/train_offer_storage/internal/api/handler"
	"github.com/go-chi/chi/v5"
	"github.com/go-chi/chi/v5/middleware"
	"net/http"
	"time"
)

type HTTPConfig struct {
	Address                    string
	PerfMetricsRefreshInterval time.Duration `config:"PERF_METRICS_REFRESH_INTERVAL,required"`
	PerfMetricsPrefix          string
}

var DefaultHTTPConfig = HTTPConfig{
	Address:                    ":9000",
	PerfMetricsRefreshInterval: 5 * time.Second,
	PerfMetricsPrefix:          "perf",
}

func RunHTTPServer(config *HTTPConfig, h *handler.HTTPHandler, logger log.Logger, metricsRegistry metrics.Registry,
	tvmHelper *tvm.TvmHelper) error {

	tm := tracing.NewTracingMiddlewareBuilder().
		WithExtractor(tracing.NewHeaderTagExtractor("request-id", "X-Request-Id")).
		Build()

	r := chi.NewRouter()
	r.Use(middleware.RequestID)
	r.Use(middleware.Recoverer)

	ping := func(w http.ResponseWriter, r *http.Request) {
		_, err := w.Write([]byte("Pong"))
		if err != nil {
			logger.Error("", log.Error(err))
		}
	}
	r.Get("/ping", ping)

	rh := r.With(tm.Handle)
	tvmHandler := tvmHelper.CheckServiceTicketHTTPMiddleware()
	if tvmHandler != nil {
		rh.Use(tvmHandler)
	}
	rh.Use(httpmetrics.New(metricsRegistry, httpmetrics.WithPathEndpoint(), httpmetrics.WithSolomonRated()))
	rh.Post("/store", h.Store)

	return http.ListenAndServe(config.Address, r)
}
