package redir

import (
	"context"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"
	"time"

	"github.com/cenkalti/backoff/v4"
	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	pb "a.yandex-team.ru/travel/proto/trains"
	"a.yandex-team.ru/travel/rasp/train_bandit_api/pkg/utils"
)

const (
	storeTrainLabelPath = "/trains/label_to_hash/"
)

type Config struct {
	Address      string        `config:"redir-address" yaml:"address"`
	Timeout      time.Duration `config:"redir-timeout" yaml:"timeout"`
	RetryTimeout time.Duration `config:"redir-retry-timeout" yaml:"retry-timeout"`
}

var DefaultConfig = Config{
	Timeout:      time.Millisecond * 300,
	RetryTimeout: time.Second * 2,
}

type Service struct {
	config  *Config
	logger  log.Logger
	client  *http.Client
	backoff *backoff.ExponentialBackOff
}

func NewService(config *Config, logger log.Logger) *Service {
	requestBackoff := &backoff.ExponentialBackOff{
		InitialInterval:     500 * time.Millisecond,
		RandomizationFactor: 0,
		Multiplier:          1.5,
		MaxInterval:         time.Second,
		MaxElapsedTime:      config.RetryTimeout,
		Clock:               backoff.SystemClock,
		Stop:                backoff.Stop,
	}
	/*
	 Request #  RetryInterval (seconds)  ElapsedTime (seconds)
	  1          0.5                     [0.5,    0.7]
	  2          0.75                    [1.25,   1.85]
	  3          1                       [2.25,   3.15]
	  4          1                       [3.25,   4.45]
	*/
	requestBackoff.Reset()

	s := &Service{
		logger:  logger,
		config:  config,
		backoff: requestBackoff,
		client: &http.Client{
			Timeout: config.Timeout,
		},
	}
	return s
}

func (s *Service) StoreLabelParams(ctx context.Context, params *pb.TLabelParams) (string, error) {
	span, _ := opentracing.StartSpanFromContext(ctx, "redir.Service.StoreLabelParams")
	defer span.Finish()

	backendURL, err := url.Parse(s.config.Address + storeTrainLabelPath)
	if err != nil {
		return "", fmt.Errorf("error parse url: %w", err)
	}
	lp, err := utils.MarshalToStr(params)
	if err != nil {
		return "", fmt.Errorf("error marshal LabelParams to string: %w", err)
	}
	args := url.Values{
		"LabelParams": {lp},
	}
	backendURL.RawQuery = args.Encode()
	backendRequest, err := http.NewRequest(http.MethodGet, backendURL.String(), nil)
	if err != nil {
		return "", fmt.Errorf("error create request: %w", err)
	}

	err = opentracing.GlobalTracer().Inject(
		span.Context(),
		opentracing.HTTPHeaders,
		opentracing.HTTPHeadersCarrier(backendRequest.Header))
	if err != nil {
		s.logger.Errorf("can not inject trace to request: %s", err.Error())
	}
	var backendResponse *http.Response
	requestFunc := func() error {
		backendResponse, err = s.client.Do(backendRequest)
		return err
	}
	err = backoff.Retry(requestFunc, s.backoff)
	if err != nil {
		return "", fmt.Errorf("error request %v: %w", backendURL, err)
	}
	defer backendResponse.Body.Close()
	responseBody, _ := ioutil.ReadAll(backendResponse.Body)
	if backendResponse.StatusCode != http.StatusOK {
		s.logger.Errorf("response code %v from %v body: %v", backendResponse.StatusCode, backendURL.String(), string(responseBody))
		return "", fmt.Errorf("http error %v: %v", backendResponse.StatusCode, string(responseBody))
	}
	return string(responseBody), nil
}
