# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import typing

from django.conf import settings
from yt.wrapper import YtClient

from common.models.transport import TransportType
from route_search.models import ZNodeRoute2


class Runner(object):

    def __init__(self, yt_proxy, yt_token):
        self.yt_client = YtClient(proxy=yt_proxy, token=yt_token)

    def dump_actual_directions(self, yt_path, dry_run=False):
        directions = self.get_directions()
        if not dry_run:
            self.write_directions(directions, yt_path)
        logging.info('All done')

    @staticmethod
    def get_directions():
        # type: () -> typing.Iterator[typing.Dict[typing.AnyStr, typing.AnyStr]]
        query = ZNodeRoute2.objects.filter(
            settlement_from__isnull=False,
            settlement_to__isnull=False,
            t_type_id__in=[TransportType.TRAIN_ID]
        ).values_list('station_from_id', 'station_to_id').distinct()

        for station_from_id, station_to_id in query:
            if station_from_id == station_to_id:
                continue
            yield dict(station_id_from=station_from_id, station_id_to=station_to_id)

    def write_directions(self, directions, yt_path):
        # type: (typing.Iterator[typing.Dict[typing.AnyStr, typing.AnyStr]], typing.AnyStr) -> None
        schema = [
            {'name': 'station_id_from', 'type': 'uint64'},
            {'name': 'station_id_to', 'type': 'uint64'},
        ]
        with self.yt_client.Transaction():
            if self.yt_client.exists(yt_path):
                self.yt_client.remove(yt_path)
            self.yt_client.create('table', yt_path, attributes={'schema': schema}, recursive=True)
            self.yt_client.write_table(yt_path, directions)


def dump_actual_directions(dry_run=False):
    Runner(settings.YT_PROXY, settings.YT_TOKEN).dump_actual_directions(settings.DUMP_ACTUAL_DIRECTIONS_OUTPUT, dry_run)
