# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from functools import partial

try:
    import pathlib
except ImportError:
    import pathlib2 as pathlib

from django.conf import settings
from django.contrib.sites.models import Site
from django.utils import six

from common.data_api.file_wrapper.config import get_wrapper_creator
from travel.rasp.library.python.sitemap.check import check_sitemaps
from travel.rasp.library.python.sitemap.generate import generate_sitemaps
from travel.rasp.library.python.sitemap.models.from_urls_list import FromUrlsListSitemap
from travel.rasp.trains.scripts.generate_sitemap.file_registry import FileType
from travel.rasp.trains.scripts.lib.fetch_routes import fetch_routes
from travel.rasp.trains.scripts.lib.fetch_searches import fetch_searches
from travel.rasp.trains.scripts.generate_sitemap.models import TrainSitemap, TravelTrainSitemap


TRAVEL_TRAIN_SITE_NAME = 'travel.yandex.ru'


def generate_sitemap(do_check_sitemaps=False, dry_run=False):
    routes = sorted(fetch_routes() | fetch_searches())
    media_root_path = pathlib.Path(settings.MEDIA_ROOT)

    trains_path = media_root_path / 'train-root'
    for site_name in settings.TRAINS_SITES:
        index_filepath, sitemap_filepaths = _write_trains(trains_path, routes, site_name)
        if do_check_sitemaps:
            check_sitemaps(index_filepath, sitemap_filepaths)

    travel_trains_path = media_root_path / 'travel-train-root'
    index_filepath, sitemap_filepaths = _write_travel_trains(travel_trains_path, routes)
    if do_check_sitemaps:
        check_sitemaps(index_filepath, sitemap_filepaths)

    if not dry_run:
        upload_sitemap(trains_path, travel_trains_path)


def _write_trains(root_path, routes, site_name):
    site = Site.objects.get(name=site_name)
    return generate_sitemaps(
        six.text_type(root_path / site.name),
        site,
        {
            'search_all_days': partial(TrainSitemap, routes),
            'trains': partial(FromUrlsListSitemap, ['/trains/sapsan']),
        }
    )


def _write_travel_trains(root_path, routes):
    travel_train_site, _created = Site.objects.get_or_create(name=TRAVEL_TRAIN_SITE_NAME,
                                                             defaults={'domain': TRAVEL_TRAIN_SITE_NAME})
    return generate_sitemaps(
        six.text_type(root_path / travel_train_site.name),
        travel_train_site,
        {
            'trains': partial(FromUrlsListSitemap, ['/trains/']),
            'search_all_days': partial(TravelTrainSitemap, routes),
            'hotels': partial(FromUrlsListSitemap, ['/hotels/']),
        }
    )


def upload_sitemap(trains_path, travel_trains_path):
    for file_type, files_path in (
        (FileType.MEDIA_TRAIN_ROOT, trains_path),
        (FileType.MEDIA_TRAVEL_TRAIN_ROOT, travel_trains_path),
    ):
        dir_wrapper = get_wrapper_creator(file_type).get_file_wrapper(six.text_type(files_path))
        dir_wrapper.upload()
