# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import typing

from common.dynamic_settings.default import conf
from common.models.geo import Settlement, Station
from common.models.transport import TransportType
from route_search.models import ZNodeRoute2


def fetch_routes():
    # type: () -> typing.Set[typing.Tuple[typing.Text, typing.Text]]
    return set(_fetch_routes())


def _fetch_routes():
    # type: () -> typing.Iterator[typing.Tuple[typing.Text, typing.Text]]
    query = ZNodeRoute2.objects.filter(
        settlement_from__isnull=False,
        settlement_to__isnull=False,
        t_type_id__in=[TransportType.TRAIN_ID]
    ).values_list('settlement_from_id', 'settlement_to_id').distinct()
    routes = set()
    settlements_ids = set()
    for settlement_from_id, settlement_to_id in query:
        if settlement_from_id == settlement_to_id:
            continue
        settlements_ids.add(settlement_from_id)
        settlements_ids.add(settlement_to_id)
        routes.add((settlement_from_id, settlement_to_id))

    settlement_by_id = Settlement.objects.in_bulk(settlements_ids)

    for settlement_from_id, settlement_to_id in routes:
        from_slug, to_slug = settlement_by_id[settlement_from_id].slug, settlement_by_id[settlement_to_id].slug
        yield from_slug, to_slug

    if conf.TRAIN_PURCHASE_SITEMAPING_ADLER_STATION:
        for route in _fetch_adler_station_routes(settlement_by_id):
            yield route


def _fetch_adler_station_routes(settlement_by_id):
    # type: (typing.Dict[int, Settlement]) -> typing.Set[typing.Tuple[typing.Text, typing.Text]]
    adler_station = Station.objects.filter(id=Station.ADLER_ID).first()
    routes = set()

    from_adler_query = ZNodeRoute2.objects.filter(
        station_from=adler_station,
        settlement_to__isnull=False,
        t_type_id__in=[TransportType.TRAIN_ID],
    ).values_list('settlement_to_id', flat=True).distinct()
    for settlement_to_id in from_adler_query:
        settlement_to = settlement_by_id.get(settlement_to_id)
        if not settlement_to:
            continue
        routes.add((adler_station.slug, settlement_to.slug))

    to_adler_query = ZNodeRoute2.objects.filter(
        station_to=adler_station,
        settlement_from__isnull=False,
        t_type_id__in=[TransportType.TRAIN_ID],
    ).values_list('settlement_from_id', flat=True).distinct()
    for settlement_from_id in to_adler_query:
        settlement_from = settlement_by_id.get(settlement_from_id)
        if not settlement_from:
            continue
        routes.add((settlement_from.slug, adler_station.slug))

    return routes
