import csv
from io import StringIO
from functools import update_wrapper
from traceback import format_exc

from django.contrib.admin import ModelAdmin
from django.contrib.admin.utils import lookup_needs_distinct
from django.db import models
from django.http import HttpResponse
from django.utils.encoding import smart_str
from django.utils.http import urlquote

FORMAT_PARAM = '_format'
COLUMNS_PARAM = '_columns'
LIMIT_PARAM = '_limit'
BOOLEAN_FORMAT = '_boolean_format'
DEFAULT_BOOLEAN_FORMAT = 'int'
IGNORED_PARAMS = (FORMAT_PARAM, COLUMNS_PARAM, LIMIT_PARAM, BOOLEAN_FORMAT)
DEFAULT_FORMAT = 'csv'
DEFAULT_ENCODING = 'utf-8'


class AllLookupModelAdmin(ModelAdmin):
    save_on_top = True

    def lookup_allowed(self, lookup, value):
        return True


class RaspExportModelAdmin(AllLookupModelAdmin):
    def get_urls(self):
        from django.conf.urls import url

        def wrap(view):
            def wrapper(*args, **kwargs):
                return self.admin_site.admin_view(view)(*args, **kwargs)
            return update_wrapper(wrapper, view)

        urlpatterns = super(RaspExportModelAdmin, self).get_urls()

        info = self.model._meta.app_label, self.model._meta.model_name

        urlpatterns = [
            url(r'^export/$', wrap(self.export_view), name='%s_%s_export' % info)
        ] + urlpatterns

        return urlpatterns

    def get_export_ignored_params(self):
        return IGNORED_PARAMS

    def export_view(self, request):
        try:
            self.request = request

            query_set = self.get_export_query_set()

            columns = self.get_columns()

            rows = self.get_rows(query_set, columns)

            formatter = self.get_export_formatter()

            return formatter(columns, rows)
        except Exception:
            return HttpResponse(u'Ошибка запроса, проверьте параметры:\n{}'.format(format_exc()),
                                content_type='text/plain; charset=utf-8')

    def get_export_query_set(self):
        ignored_params = self.get_export_ignored_params()

        lookup_params = self.request.GET.copy()

        for ignored in ignored_params:
            if ignored in lookup_params:
                del lookup_params[ignored]

        # Normalize the types of keys
        for key, value in lookup_params.items():
            if not isinstance(key, str):
                # 'key' will be used as a keyword argument later, so Python
                # requires it to be a string.
                del lookup_params[key]
                lookup_params[smart_str(key)] = value

        use_distinct = False

        filter_params = {}

        add_filters = []

        for key, value in lookup_params.items():
            filter_param = prepare_lookup_value(key, value)

            if isinstance(filter_param, models.Q):
                add_filters.append(filter_param)
            else:
                filter_params[key] = filter_param

            use_distinct = (use_distinct or lookup_needs_distinct(self.opts, key))

        query_set = self.get_queryset(self.request).filter(**filter_params)

        for add_filter in add_filters:
            query_set = query_set.filter(add_filter)

        return query_set

    def get_columns(self):
        fields = list(filter(None, self.request.GET.get(COLUMNS_PARAM, '').split(',')))

        return fields or [f.get_attname() for f in self.opts.local_fields]

    def get_values(self, obj, columns):
        return [self.get_column_value(obj, column) for column in columns]

    def get_column_value(self, obj, column):
        current_value = obj
        for attr_name in column.split('__'):
            current_value = self.get_attr_or_field_value(current_value, attr_name)

        return self.prepare_to_export(current_value)

    def get_attr_or_field_value(self, obj, attr_name):
        attr = getattr(obj, attr_name)
        if callable(attr):
            return attr()
        else:
            return attr

    def get_rows(self, query_set, columns):
        limit = self.request.GET.get(LIMIT_PARAM, None)
        if limit:
            query_set = query_set[:int(limit)]

        for obj in query_set:
            yield self.get_values(obj, columns)

    def prepare_to_export(self, value):
        if value is None:
            return u""
        elif isinstance(value, models.Model):
            return str(value.id)
        elif isinstance(value, bool):
            if self.request.GET.get(BOOLEAN_FORMAT, DEFAULT_BOOLEAN_FORMAT) == 'int':
                return str(int(value))
            else:
                return str(value)
        else:
            return str(value)

    def get_export_formatter(self):
        format = self.request.GET.get(FORMAT_PARAM, DEFAULT_FORMAT)
        return getattr(self, "%s_export_formatter" % format)

    def csv_export_formatter(self, columns, rows, delimiter=';', ext="csv"):
        stream = StringIO()
        writer = csv.writer(stream, delimiter=delimiter)
        writer.writerow(columns)
        for row in rows:
            writer.writerow(row)

        response = HttpResponse(stream.getvalue().encode(), content_type='text/csv')

        filename = self.model.__name__.lower() + 's.' + ext
        response['Content-Disposition'] = 'attachment; filename={}'.format(urlquote(filename))

        return response

    def tsv_export_formatter(self, columns, rows):
        return self.csv_export_formatter(columns, rows, delimiter="\t", ext="tsv")


def prepare_lookup_value(key, value):
    """
    Returns a lookup value prepared to be used in queryset filtering.
    """
    # if key ends with __in, split parameter into separate values
    if key.endswith('__in'):
        value = value.split(',')
    # if key ends with __isnull, special case '' and false
    if key.endswith('__isnull'):
        if value.lower() in ('', '0', 'false'):
            value = False
        else:
            value = True

    if key.endswith('__empty'):
        key = key.replace('__empty', "")

        is_empty = models.Q(**{key: ""}) | models.Q(**{key + '__isnull': True})

        if value.lower() in ('', '0', 'false'):
            return ~ is_empty
        else:
            return is_empty

    return value
