from django.core.validators import RegexValidator
from django.db import models

from .utils import CodeCharField, TrimmedCharField, RegExpCharField, RegExpTextField

CLASS_CHOICES = [
    ('compartment', 'купе'),
    ('suite', 'СВ/люкс'),
    ('sitting', 'сидячий'),
    ('platzkarte', 'плацкарт'),
    ('soft', 'мягкий'),
    ('common', 'общий'),
]


class StationExpressAlias(models.Model):
    """Название станции в системе Экспресс"""
    station = models.IntegerField(verbose_name='Станция', null=False, blank=False)
    alias = CodeCharField(verbose_name='Название', max_length=50, unique=True, null=False, blank=False, db_index=True)

    def __str__(self):
        return self.alias

    class Meta:
        verbose_name = 'название в системе Экспресс'
        verbose_name_plural = 'названия в системе Экспресс'


class CoachSchema(models.Model):
    """Схема вагона"""
    name = TrimmedCharField(verbose_name='Название', max_length=100, blank=False, default=None)
    all_places = models.TextField(verbose_name='Все места (через запятую)', blank=True, default='')
    details = models.TextField(verbose_name='Дополнительная информация о вагоне и местах', blank=True, default='')
    svg_schema = models.TextField(verbose_name='Размеченная схема вагона (содержание svg файла)', blank=True, default='')
    two_storey = models.BooleanField(verbose_name='Двухэтажный вагон', default=False)

    def __str__(self):
        return self.name

    class Meta:
        verbose_name = 'схема вагонов'
        verbose_name_plural = 'схемы вагонов'


class CoachBinding(models.Model):
    """Привязка схемы к вагону"""
    klass = models.CharField(verbose_name='Категория вагона', max_length=100,
                             choices=CLASS_CHOICES, blank=True, default='')
    service_class = RegExpCharField(verbose_name='Класс обслуживания', max_length=100, blank=True, default='')
    international_service_class = RegExpCharField(verbose_name='Международный класс обслуживания', max_length=100,
                                                  blank=True, default='')
    start_settlement_id = models.IntegerField(verbose_name='Город отправления поезда', null=True, blank=True)
    end_settlement_id = models.IntegerField(verbose_name='Город прибытия поезда', null=True, blank=True)
    coach_subtype_code = TrimmedCharField(verbose_name='Подтип вагона', max_length=100, blank=True, default='',
                                          help_text='Можно указать несколько через запятую')
    road = RegExpCharField(verbose_name='Дорога принадлежности вагона', max_length=255, blank=True, default='',
                           help_text='или государство принадлежности')
    train_number = RegExpTextField(verbose_name='Номер поезда', blank=True, default='')
    coach_number = RegExpCharField(verbose_name='Номер вагона', max_length=255, blank=True, default='')
    priority = models.IntegerField(verbose_name='Приоритет', blank=False, default=10)
    schema = models.ForeignKey(CoachSchema, models.CASCADE, verbose_name='схема', null=False, blank=False)
    direction_confirmed = models.BooleanField(verbose_name='Направление движения подтверждено', default=False)

    def __str__(self):
        return "{} {} {} {}".format(self.klass, self.train_number, self.coach_number, self.priority)

    class Meta:
        verbose_name = 'привязка схемы вагона'
        verbose_name_plural = 'привязки схем вагонов'


class ServiceClass(models.Model):
    """Класс обслуживания"""
    name = models.CharField(verbose_name='Название для админки', max_length=1024, blank=True, default='')
    code = CodeCharField(verbose_name='Коды классов обслуживания', max_length=100, blank=False,
                         help_text='Можно указать несколько через запятую')
    title = TrimmedCharField(verbose_name='Название для отображения (короткое)', max_length=100, blank=False)
    description = models.TextField(verbose_name='Описание класса обслуживания для отображения', blank=True, default='')
    coach_category = models.CharField(verbose_name='Категория вагона', max_length=20,
                                      choices=CLASS_CHOICES, blank=True, default='')
    coach_owner = models.CharField(verbose_name='Владельцы вагонов', max_length=100, blank=True, default='',
                                   help_text='Можно указать несколько через запятую')
    brand_title = models.CharField(verbose_name='Название бренда', max_length=100, blank=True, default='')
    is_firm_coach = models.NullBooleanField(verbose_name='Вагон является фирменным', null=True, blank=True, default=None)
    train_number = RegExpCharField(verbose_name='Номер поезда', max_length=1024, blank=True, default='',
                                   help_text='Регулярное выражение')
    two_storey = models.NullBooleanField(verbose_name='Двухэтажный вагон', null=True, blank=True, default=None)
    start_date = models.DateField(verbose_name='Дата начала действия класса обслуживания',
                                  null=True, blank=True, default=None)
    end_date = models.DateField(verbose_name='Дата окончания действия класса обслуживания',
                                null=True, blank=True, default=None)

    class Meta:
        verbose_name = 'Класс обслуживания'
        verbose_name_plural = 'Классы обслуживания'


class Facility(models.Model):
    """Удобство"""
    name = models.CharField(verbose_name='Название для админки', max_length=1024, blank=True, default='')
    code = CodeCharField(verbose_name='Код', max_length=100, blank=False, unique=True, validators=[
        RegexValidator('[A-Za-z][_A-Za-z0-9]*', 'Код может содержать только латинские буквы, цифры и знак "_"')
    ])
    im_code = CodeCharField(verbose_name='Код в системе IM', max_length=100, blank=False, unique=True)
    title = TrimmedCharField(verbose_name='Название для отображения', max_length=100, blank=False)
    for_place = models.BooleanField(verbose_name='Удобство связанное с местом', blank=True, default=False)
    order = models.IntegerField(verbose_name='Порядок отображения', blank=False, default=0)

    class Meta:
        verbose_name = 'Удобство'
        verbose_name_plural = 'Удобства'
