import re

from django import forms
from django.db import models
from django.utils.encoding import force_text


class CodeCharField(models.CharField):
    """
    Обращает пустую строку в None, чтобы было удобнее обращаться с unique=True
    """
    def pre_save(self, instance, add):
        value = models.CharField.pre_save(self, instance, add)

        if value is not None:
            value = value.strip() or None
            setattr(instance, self.attname, value)
        return value


class TrimmedCharField(models.CharField):
    def pre_save(self, instance, add):
        value = models.CharField.pre_save(self, instance, add)

        if isinstance(value, str):
            value = value.strip()
            setattr(instance, self.attname, value)
        return value


class RegExpFormField(forms.CharField):
    def clean(self, value):
        value = super(RegExpFormField, self).clean(value)
        try:
            value = value.strip()
            re.compile(value, re.U)
        except re.error as e:
            raise forms.ValidationError('Ошибка в регулярном выражении: {}'.format(force_text(e)))
        return value


class RegExpCharField(models.CharField):
    def formfield(self, **kwargs):
        defaults = {'form_class': RegExpFormField}
        defaults.update(kwargs)
        return super(RegExpCharField, self).formfield(**defaults)


class RegExpTextField(models.TextField):
    def formfield(self, **kwargs):
        defaults = {'form_class': RegExpFormField}
        defaults.update(kwargs)
        return super(RegExpTextField, self).formfield(**defaults)
