#!/usr/bin/env python
# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import argparse
import json
import os
import sys
from subprocess import check_call, CalledProcessError

from utils import patch_stdstreams
from utils.process import check_output_stream, check_output_stripped
from utils.version import Version, ReleaseTypes
from utils.git import get_current_branch_remote, get_current_branch_name, get_remote_info


DESCRIPTION = """\
Скрипт для бекмержей
"""

GITHUB_API_URL = 'https://api.github.yandex-team.ru/'


def send_pr(title, from_branch, to_branch, **kwargs):
    data = json.dumps({
        'title': title, 'body': kwargs.get('body', ''),
        'head': from_branch, 'base': to_branch
    })
    remote_info = get_remote_info()
    request_url = '{}repos/{}/{}/pulls'.format(GITHUB_API_URL, remote_info['owner'], remote_info['repo'])
    result = check_output_stripped(['curl', '-s', '--user', '{}:{}'.format(kwargs['github_user'], kwargs['github_password']),
                                    '--data', data, request_url])
    result = json.loads(result)
    if result.get('errors'):
        print(result)
        return False
    else:
        print(result['html_url'])
        return True


def handle_backmerge(args):
    try:
        current_branch = get_current_branch_name()
        check_call(['git', 'fetch', 'origin'])
        check_call(['git', 'checkout', args.to_branch])
        check_call(['git', 'merge', args.from_branch])
        check_call(['git', 'push', 'origin', args.to_branch])
        check_call(['git', 'checkout', current_branch])
    except CalledProcessError as exc:
        print('Ошибка при попытке сделать backmerge {}, создаем PR'.format(repr(exc)))
        send_pr('Backmerge {}'.format(args.from_branch), args.from_branch, args.to_branch, **{
            'body': args.body,
            'github_user': args.github_user,
            'github_password': args.github_password,
        })
        sys.exit(1)


def handle_send_pr(args):
    sent = send_pr(args.title, args.from_branch, args.to_branch, **{
        'body': args.body,
        'github_user': args.github_user,
        'github_password': args.github_password,
    })
    if not sent:
        sys.exit(1)


if __name__ == '__main__':
    patch_stdstreams()
    parser = argparse.ArgumentParser(DESCRIPTION)
    
    parser.add_argument('--github-user', default='robot-rasp')
    parser.add_argument('--github-password', default=os.environ.get('ROBOT_RASP_PASSWORD'))

    subparsers = parser.add_subparsers()

    backmerge_parser = subparsers.add_parser('backmerge')
    backmerge_parser.add_argument('--from-branch', default=get_current_branch_name())
    backmerge_parser.add_argument('--to-branch', default='dev')
    backmerge_parser.add_argument('--body', default='')
    backmerge_parser.set_defaults(handler=handle_backmerge)

    send_pr_parser = subparsers.add_parser('send-pr')
    send_pr_parser.add_argument('--from-branch', default=get_current_branch_name())
    send_pr_parser.add_argument('--to-branch', required=True)
    send_pr_parser.add_argument('--title', required=True)
    send_pr_parser.add_argument('--body', default='')
    send_pr_parser.set_defaults(handler=handle_send_pr)

    args = parser.parse_args()
    args.handler(args)
