# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import re
import sys
from contextlib import contextmanager
from subprocess import CalledProcessError

from utils.process import check_output_stream, check_output_stripped
from utils.version import BadTagName, Version
from utils.version_tag_mark import version_tag_mark


UPSTREAM_REPO_GROUP = 'git@github.yandex-team.ru:rasp/'
RE_COMMIT_LINE = re.compile(r'^commit (?P<hash>[0-9a-f]+)(?: \((?P<refs>.*)\))?$')


class Commit(object):
    def __init__(self):
        self.hash = None
        self.tags = []
        self.version = None
        self.author = None
        self.email = None
        self.header = None
        self.body = None
        self.is_merge = False

    @property
    def short_hash(self):
        return self.hash[:8]

    @classmethod
    def parse_commit(cls, commit_lines):
        commit = Commit()
        commit._parse_commit_line(commit_lines[0])
        commit._parse_author(commit_lines)
        commit._parse_comment(commit_lines)
        commit._parse_merge(commit_lines)

        return commit

    def _parse_commit_line(self, commit_line):
        match = RE_COMMIT_LINE.match(commit_line)
        if not match:
            raise Exception('Bad commit')
        groups = match.groupdict(default='')
        self.hash = groups['hash']

        for ref in groups['refs'].split(', '):
            if ref.startswith('tag: '):
                self.tags.append(ref.replace('tag: ', ''))

        for tag in self.tags:
            if tag.startswith(version_tag_mark.get()):
                try:
                    self.version = Version.from_tag_name(tag)
                    break
                except BadTagName:
                    print('Bad tag name {}'.format(tag), file=sys.stderr)

    def _parse_author(self, commit_lines):
        for line in commit_lines:
            if line.startswith('Author: '):
                break
        else:
            return

        line = line[len('Author: '):]
        self.author, email = line.split(' <', 1)
        self.email = email.rstrip('>')

    def _parse_comment(self, commit_lines):
        body_lines = []
        for line in commit_lines:
            if line.startswith('    '):
                if self.header:
                    body_lines.append(line.lstrip())
                else:
                    self.header = line.lstrip()
        if body_lines:
            self.body = '\n'.join(body_lines)

    def _parse_merge(self, commit_lines):
        for line in commit_lines:
            if line.startswith('Merge: '):
                self.is_merge = True
                break


def _iter_commits_lines(stream):
    commit = []
    for line in stream:
        line = line.rstrip()
        if line.startswith('commit'):
            if commit:
                yield commit
                commit = []
            commit.append(line)
        else:
            commit.append(line)

    if commit:
        yield commit


def _iter_commits(stream):
    for commit_lines in _iter_commits_lines(stream):
        yield Commit.parse_commit(commit_lines)


@contextmanager
def build_commit_iter(*gitargs):
    git_cmd = ['git', 'log', '--topo-order', '--raw', '--decorate=short'] + list(gitargs)
    with check_output_stream(git_cmd) as stream:
        yield _iter_commits(stream)


def get_config_param(param):
    try:
        return check_output_stripped(['git', 'config', param])
    except CalledProcessError as ex:
        if ex.returncode == 1:  # параметр не определен
            return None
        else:
            raise


def get_current_branch_name():
    ref = check_output_stripped(['git', 'symbolic-ref', 'HEAD'])
    assert ref.startswith('refs/heads/')
    return ref[len('refs/heads/'):]


def get_current_branch_remote():
    current_branch = get_current_branch_name()
    current_branch_remote = get_config_param('branch.{}.remote'.format(current_branch))
    return current_branch_remote


REMOTE_URL_RE = re.compile(r'\S+[:/](?P<owner>[^/]+)/(?P<repo>[^/]+)$')


def get_remote_info(remote='origin'):
    remote_url = check_output_stripped(['git', 'config', '--get', 'remote.{}.url'.format(remote)])

    remote_info = REMOTE_URL_RE.match(remote_url).groupdict()
    if remote_info['repo'].endswith('.git'):
        remote_info['repo'] = remote_info['repo'][:-4]
    return remote_info
