# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from collections import namedtuple

from utils.version_tag_mark import version_tag_mark


class BadTagName(Exception):
    pass


class ReleaseTypes(object):
    MAJOR = 'major'
    MINOR = 'minor'
    FIX = 'fix'
    EXPERIMENTAL = 'experimental'

    @classmethod
    def choices(cls):
        return [cls.MAJOR, cls.MINOR, cls.FIX, cls.EXPERIMENTAL]


class Version(namedtuple('Version', 'major, minor, fix, rest')):
    def inc_major_version(self):
        return self._replace(major=self.major + 1, minor=0, fix=0, rest=None)

    def inc_minor_version(self):
        return self._replace(major=self.major, minor=self.minor + 1, fix=0, rest=None)

    def inc_fix_version(self):
        return self._replace(major=self.major, minor=self.minor, fix=self.fix + 1, rest=None)

    def inc_experimental_version(self, experiment_tag=None):
        if not self.rest:
            if not experiment_tag:
                raise Exception('You must pass experiment tag for the first experiment version')
            rest = '{}{:03d}'.format(experiment_tag, 1)
        else:
            if experiment_tag and self.rest[:-3] != experiment_tag:
                raise Exception('You must not change experiment tag')
            rest = '{}{:03d}'.format(self.rest[:-3], int(self.rest[-3:], 10) + 1)

        return self._replace(major=self.major, minor=self.minor, fix=self.fix, rest=rest)

    def inc_version(self, release_type=ReleaseTypes.MINOR, experiment_tag=None):
        if release_type == ReleaseTypes.MAJOR:
            return self.inc_major_version()
        elif release_type == ReleaseTypes.MINOR:
            return self.inc_minor_version()
        elif release_type == ReleaseTypes.FIX:
            return self.inc_fix_version()
        elif release_type == ReleaseTypes.EXPERIMENTAL:
            return self.inc_experimental_version(experiment_tag)
        else:
            raise Exception('Bad release type')

    def to_tag_name(self):
        tag = version_tag_mark.get() + '-{major}.{minor}.{fix}'.format(**self._asdict())
        if self.rest:
            tag += '.{}'.format(self.rest)
        return tag

    @classmethod
    def from_tag_name(cls, tag_name):
        version = tag_name.replace(version_tag_mark.get() + '-', '')
        parts = version.split('.', 3)
        try:
            major, minor, fix = map(int, parts[:3])
        except ValueError:
            raise BadTagName(tag_name)
        rest = parts[3] if len(parts) == 4 else None

        return cls(major=major, minor=minor, fix=fix, rest=rest)
