#!/usr/bin/env python
# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import argparse
import os
import sys
from subprocess import check_call

from utils import patch_stdstreams
from utils.git import UPSTREAM_REPO_GROUP, build_commit_iter, get_config_param, get_current_branch_remote
from utils.process import check_output_stream
from utils.version import Version, ReleaseTypes
from utils.version_tag_mark import version_tag_mark


DESCRIPTION = """\
Скрипт для работы установки тегов и построения истории коммитов
"""


def search_last_version():
    """
    :return: (version, is_current)
    :rtype: tuple[utils.version.Version, bool]
    """
    with build_commit_iter() as commit_iter:
        for index, commit in enumerate(commit_iter):
            if commit.version:
                return commit.version, index == 0

    return None, False


def _check_push_remote(remote):
    remote_url = get_config_param('remote.{}.url'.format(remote)) or ''
    if not (remote_url.startswith(UPSTREAM_REPO_GROUP) or remote.startswith(UPSTREAM_REPO_GROUP)):
        print('Warning! Пушим тег не в общий репозиторий, сборка может сломаться', file=sys.stderr)


def version_is_pushed(version, remote):
    with check_output_stream(['git', 'ls-remote', '--tags', remote]) as stream:
        for line in stream:
            if 'refs/tags/{}'.format(version.to_tag_name()) in line:
                return True
    return False


def handle_new_tag_command(args):
    version_tag_mark.set(args.version_tag_mark)
    last_version, is_current = search_last_version()

    if is_current:
        print('Текущий коммит уже имеет версию', last_version.to_tag_name())
        version = last_version
        # Выходим сразу, чтобы new-tag отрабатывал на refs/tags/*
        return

    remote = args.remote or get_current_branch_remote()
    _check_push_remote(remote)

    last_version = last_version or Version(0, 0, 0, None)
    version = last_version.inc_version(args.release_type, args.experiment_tag)
    print('Следующая версия {}'.format(version.to_tag_name()))

    if not args.dry:
        print('Ставим тег', version.to_tag_name())
        check_call(['git', 'tag', version.to_tag_name()])

    if is_current and version_is_pushed(version, remote):
        print('Эта версия уже запушена')
        return

    if not args.dry:
        print('Пушим тег {} в {}'.format(version.to_tag_name(), remote))
        check_call(['git', 'push', '-v', remote, 'tag', version.to_tag_name()])


def _get_start_from_object(args):
    if args.start_from:
        return args.start_from

    version_count = 0
    commit = None
    with build_commit_iter() as iterator:
        for commit in iterator:
            if commit.version:
                version_count += 1
            if version_count >= args.max_versions + 1:
                break
    if not commit:
        raise Exception('Нет коммитов в текущей ветке')

    if commit.tags:
        return commit.tags[0]
    else:
        return commit.hash


class SimpleFormatter(object):
    def format_version_line(self, commit):
        return '### {}'.format(commit.version.to_tag_name())

    def format_commit_line(self, commit):
        return ' * [{commit.author}] "{commit.header}" {commit.short_hash}'.format(commit=commit)


def format_commit(commit, formatter):
    if commit.version:
        yield ''
        yield formatter.format_version_line(commit)
        yield ''

    if not commit.is_merge:
        yield formatter.format_commit_line(commit)


def handle_changelog_command(args):
    version_tag_mark.set(args.version_tag_mark)
    start_from_object = _get_start_from_object(args)
    changelog = []
    with build_commit_iter('{}..'.format(start_from_object)) as iterator:
        for commit in iterator:
            changelog.extend(format_commit(commit, SimpleFormatter()))

    # чистим пустые строки в начале
    while changelog and not changelog[0]:
        changelog.pop(0)

    print('\n'.join(changelog))


def handler_get_version(args):
    version_tag_mark.set(args.version_tag_mark)
    with build_commit_iter() as iterator:
        commit = next(iterator)

    if commit.version:
        print(commit.version.to_tag_name())


def handler_new_release(args):
    check_call(['git', 'fetch', 'origin'])
    check_call(['git', 'checkout', args.release_branch])
    check_call(['git', 'merge', 'origin/dev'])
    if args.run_tests and os.path.exists('Makefile'):
        check_call(['make', 'test'])

    check_call(['git', 'push', 'origin', args.release_branch])


if __name__ == '__main__':
    patch_stdstreams()
    parser = argparse.ArgumentParser(DESCRIPTION)
    subparsers = parser.add_subparsers()

    new_tag_parser = subparsers.add_parser('new-tag')
    new_tag_parser.add_argument('--dry', default=False, action='store_true')
    new_tag_parser.add_argument('--remote', default=None)
    new_tag_parser.add_argument('--release-type', default='minor', choices=ReleaseTypes.choices())
    new_tag_parser.add_argument('--experiment-tag')
    new_tag_parser.add_argument('--version-tag-mark', default='release', type=str, help='Release tag prefix')
    new_tag_parser.set_defaults(handler=handle_new_tag_command)

    changelog_parser = subparsers.add_parser('changelog')
    changelog_parser.add_argument('--start-from', default=None, type=str, help='Tag, branch or commit')
    changelog_parser.add_argument('--max-versions', default=30, type=int)
    changelog_parser.add_argument('--version-tag-mark', default='release', type=str, help='Release tag prefix')
    changelog_parser.set_defaults(handler=handle_changelog_command)

    get_version_parser = subparsers.add_parser('get-version')
    get_version_parser.set_defaults(handler=handler_get_version)
    get_version_parser.add_argument('--version-tag-mark', default='release', type=str, help='Release tag prefix')

    new_release_parser = subparsers.add_parser('new-release')
    new_release_parser.add_argument('--run-tests', dest='run_tests', action='store_false')
    new_release_parser.add_argument('--release-branch', default='release', type=str,
                                    help='Branch, from which release will be built')
    new_release_parser.set_defaults(handler=handler_new_release)

    args = parser.parse_args()
    args.handler(args)
