package main

import (
	"context"
	"fmt"
	"os"
	"time"

	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/rasp/wizards/go/wizard_proxy_api/internal/application"
	"a.yandex-team.ru/travel/rasp/wizards/go/wizard_proxy_api/internal/logging"
	"a.yandex-team.ru/travel/rasp/wizards/go/wizard_proxy_api/internal/server"
	"a.yandex-team.ru/travel/rasp/wizards/go/wizard_proxy_api/internal/storage"
)

const (
	RaspWPATesting    = ""
	RaspWPAProduction = ""
)

func getSecUID() string {
	envType := os.Getenv("YENV_TYPE")
	secUID := ""
	switch envType {
	case "production":
		secUID = RaspWPAProduction
	case "development":
		fallthrough
	case "testing":
		secUID = RaspWPATesting
	}
	return secUID
}

func main() {
	maxprocs.AdjustAuto()
	configLoader := configuration.NewDefaultConfitaLoader()

	perfMetricsRefreshInterval, _ := time.ParseDuration("10m")
	serverConfig := server.Config{
		Port:                       "9999",
		PerfMetricsRefreshInterval: perfMetricsRefreshInterval,
		PerfMetricsPrefix:          "wizard_proxy_api_go",
	}

	err := configLoader.Load(context.Background(), &serverConfig)
	if err != nil {
		fmt.Println("Config Loader load server config error: ", err)
	}

	appConfig := application.Config{
		ReturnSingleCityIfOneCityAndNoOutsideNonBusStations: true,
		SearchPointLimit: 100,
	}

	err = configLoader.Load(context.Background(), &appConfig)
	if err != nil {
		fmt.Println("Config Loader load app config error: ", err)
	}

	logConfig := logging.LogConfig{
		LogPath: "./logs.log",
		Level:   "INFO",
	}

	err = configLoader.Load(context.Background(), &logConfig)
	if err != nil {
		fmt.Println("Config Loader load logger config error: ", err)
	}

	logger, err := logging.New(logConfig)
	if err != nil {
		fmt.Println("failed to create logger, err:", err)
		return
	}
	defer logger.L.Sync()

	s := storage.Storage{}
	s.LoadRepositories(logger)

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	app, err := application.NewApplication(ctx, &appConfig, &s, logger)
	if err != nil {
		logger.Error(fmt.Sprintf("Cannot create new application: %v", err))
	}

	srv, err := server.New(ctx, &serverConfig, app)
	if err != nil {
		logger.Error(fmt.Sprintf("failed to create Server: %v", err))
		return
	}
	defer srv.Close()
	if err = srv.ListenAndServe(); err != nil {
		logger.Error(fmt.Sprintf("failed to run Server: %v", err))
		return
	}
}
