package direction

import (
	"errors"

	"a.yandex-team.ru/travel/rasp/wizards/go/wizard_proxy_api/internal/application"
	"a.yandex-team.ru/travel/rasp/wizards/go/wizard_proxy_api/internal/geomodel"
	"a.yandex-team.ru/travel/rasp/wizards/go/wizard_proxy_api/internal/geosearch"
	"a.yandex-team.ru/travel/rasp/wizards/go/wizard_proxy_api/internal/storage"
)

type FindPointsParameters struct {
	Cfg     *application.Config
	Storage *storage.Storage

	TransportCode      string
	DeparturePointName string
	ArrivalPointName   string

	ClientGeoid              int
	DepartureSettlementGeoid int
	ArrivalSettlementGeoid   int
}

func FindPoints(params *FindPointsParameters) (geomodel.Point, geomodel.Point, error) {
	isSuburbanSearch := (params.TransportCode == "suburban")
	tType := geomodel.TransportCodeFromString(params.TransportCode)

	var searcher geosearch.PointSearcher
	if isSuburbanSearch {
		searcher = geosearch.BasePointSearch{
			Cfg:     params.Cfg,
			Storage: params.Storage,
		}
	} else {
		searcher = geosearch.SuburbanPointSearch{
			Cfg:     params.Cfg,
			Storage: params.Storage,
		}
	}

	departureSettlement, err := params.Storage.GetSettlementByGeoID(params.DepartureSettlementGeoid, true)

	var departurePointList *geosearch.PointList
	if err == nil {
		departurePointList, err = geosearch.NewPointList(departureSettlement, true)
		if err != nil {
			return nil, nil, errors.New("cannot create point list for recovered departure point")
		}
	} else {
		departurePointList, err = geosearch.FindPointList(searcher, params.DeparturePointName, tType, "", "")
		if err != nil {
			return nil, nil, errors.New("cannot recover departure point")
		}
	}

	arrivalSettlement, err := params.Storage.GetSettlementByGeoID(params.ArrivalSettlementGeoid, true)

	var arrivalPointList *geosearch.PointList
	if err == nil {
		arrivalPointList, err = geosearch.NewPointList(arrivalSettlement, true)
		if err != nil {
			return nil, nil, errors.New("cannot create point list for recovered arrival point")
		}
	} else {
		arrivalPointList, err = geosearch.FindPointList(searcher, params.ArrivalPointName, tType, "", "")
		if err != nil {
			return nil, nil, errors.New("cannot recover arrival point")
		}
	}

	clientSettlement, _ := params.Storage.GetSettlementByGeoID(params.ClientGeoid, true)

	_, _, err = geosearch.ProcessPointsLists(
		params.Storage,
		departurePointList,
		arrivalPointList,
		clientSettlement,
		false,
		isSuburbanSearch,
		false,
		false,
	)
	if err != nil {
		return nil, nil, errors.New("cannot find any route for current points")
	}

	departurePointList, arrivalPointList, err = geosearch.ProcessPointsLists(
		params.Storage,
		departurePointList,
		arrivalPointList,
		clientSettlement,
		true,
		isSuburbanSearch,
		false,
		false,
	)
	if err != nil {
		return nil, nil, errors.New("cannot find any route for current points")
	}

	return departurePointList.MainPoint, arrivalPointList.MainPoint, nil
}
