package geomodel

type TransportType byte

const (
	ETransportNone TransportType = iota
	ETransportTrain
	ETransportPlane
	ETransportBus
	ETransportRiver
	ETransportSea
	ETransportSuburban
	ETransportHelicopter
	ETransportUrban
	ETransportPseudoGortrans
	ETransportBlablacar
	ETransportWater
	ETransportAeroexpress // TODO It's not from common.model. I add it by myself, because in DB 'aeroex' is different from 'suburban'
)

var WaterTTypeCodes = map[TransportType]bool{ETransportRiver: true, ETransportSea: true, ETransportWater: true}
var TransportTypeByTextName = map[string]TransportType{
	"train":      ETransportTrain,
	"plane":      ETransportPlane,
	"bus":        ETransportBus,
	"river":      ETransportRiver,
	"sea":        ETransportSea,
	"suburban":   ETransportSuburban,
	"helicopter": ETransportHelicopter,
	"urban":      ETransportUrban,
	"gortran":    ETransportPseudoGortrans, // TODO: check correct value
	"blablacar":  ETransportBlablacar,      // TODO: check correct value
	"water":      ETransportWater,
	"aeroex":     ETransportAeroexpress,
	// TODO: db contains 'tablo'
	// TODO: in db type_choices may be cutted, for example for id=43 '..helicopte'
}

func (t TransportType) String() string {
	return [...]string{"Train", "Plane", "Bus", "River", "Sea", "Suburban", "Helicopter", "Urban", "PseudoGortrans", "Blablacar", "Water", "Aeroexpress"}[t]
}

func ContainWaterTType(types map[TransportType]bool) bool {
	for t := range WaterTTypeCodes {
		_, found := types[t]
		if found {
			return true
		}
	}
	return false
}

func IsWaterTType(t TransportType) bool {
	_, found := WaterTTypeCodes[t]
	return found
}

func TransportCodeFromString(sType string) TransportType { // TODO are the same http query parameters and DB string parameters?
	tType, found := TransportTypeByTextName[sType]
	if !found {
		return ETransportNone
	}
	return tType
}
