package geosearch

import (
	"a.yandex-team.ru/travel/rasp/wizards/go/wizard_proxy_api/internal/geomodel"
	"a.yandex-team.ru/travel/rasp/wizards/go/wizard_proxy_api/internal/storage"
)

const defaultBlockSize = 10

// Проверка соответствия станций найденным городам, фильтрация скрытых и неважных станций
func getSuitableStationFilter(settlements []*geomodel.Settlement) func(*storage.Storage, *geomodel.Station) bool {
	var settlementTitles map[int]string
	for _, s := range settlements {
		settlementTitles[s.GetID()] = normalize(s.GetTitle())
	}

	filterFunction := func(s *storage.Storage, station *geomodel.Station) bool {
		// Недостаточно важные
		if station.GetMajorityID() >= geomodel.StationMajorityNotInSearchID {
			return false
		}

		// Скрытые
		if station.IsHidden() {
			return false
		}

		// не предлагать для уточнения на выбор - город и одноименную станцию внутри города
		// (RASP-1874)
		// если города нет, то пропускаем
		id, err := station.GetSettlementID()
		if err != nil {
			return true
		}

		title, found := settlementTitles[id]
		if found && title == normalize(station.GetTitle()) {
			return false
		}
		return true
	}

	return filterFunction
}

// Проверка TransportType точки
func conformTType(point geomodel.Point, tType geomodel.TransportType) bool {
	if tType == geomodel.ETransportNone {
		return true
	}
	typeSet, err := point.GetTypeChoicesSet()
	if err == nil {
		_, found := typeSet[tType]
		if found {
			return true
		}
		if tType == geomodel.ETransportSuburban {
			_, found := typeSet[geomodel.ETransportAeroexpress]
			if found {
				return true
			}
		}
		if tType == geomodel.ETransportWater && geomodel.ContainWaterTType(typeSet) {
			return true
		}
	}
	if tType == geomodel.ETransportAeroexpress && point.GetType() == geomodel.ESettlement {
		return true
	}
	if point.GetType() == geomodel.EStation {
		pointTType := geomodel.ETransportNone // TODO: pointTType := *point.(geomodel.Station).GetTransportType()
		if geomodel.IsWaterTType(tType) && geomodel.IsWaterTType(pointTType) {
			return true
		}
		return pointTType == tType
	}
	return false // unexpected point type, remove it // TODO may be log it?
}

// Ищет страны в NameSearch словарях и дописывает в конец списка
func (pointSearch BasePointSearch) AppendCountries(title string, matchType storage.NameSearchIndexDict, points []geomodel.Point, seenIds map[int]bool) ([]geomodel.Point, map[int]bool) {
	countries, err := pointSearch.Storage.NameSearcher.FindCountries(matchType, title)
	if err == nil {
		for _, p := range countries {
			id := p.GetID()
			_, found := seenIds[id]
			if !found {
				seenIds[id] = true
				points = append(points, p)
			}
		}
	}
	return points, seenIds
}

// Ищет города в NameSearch словарях, сортирует и дописывает в конец списка
func (pointSearch BasePointSearch) AppendSettlements(title string, tType geomodel.TransportType, matchType storage.NameSearchIndexDict, points []geomodel.Point, seenIds map[int]bool) ([]geomodel.Point, map[int]bool, []*geomodel.Settlement) {
	settlements, err := findSettlementsFiltered(pointSearch.Storage, matchType, title, func(s *geomodel.Settlement) bool { return !s.IsHidden() })
	if err != nil {
		settlements = make([]*geomodel.Settlement, 0)
	}
	filteredSettlements := make([]*geomodel.Settlement, defaultBlockSize)
	for _, p := range settlements {
		id := p.GetID()
		_, found := seenIds[id]
		if !found && conformTType(p, tType) {
			seenIds[id] = true
			filteredSettlements = append(filteredSettlements, p)
		}
	}
	sortedSettlements := sortedSettlements(filteredSettlements)
	for _, s := range sortedSettlements {
		points = append(points, geomodel.Point(s))
	}
	return points, seenIds, settlements
}

// Ищет станции в NameSearch словарях, сортирует и дописывает в конец списка
// Последний параметр в ответе - необходимость сразу вернуть список, без добавления других блоков
func (pointSearch BasePointSearch) AppendStations(title string, tType geomodel.TransportType, matchType storage.NameSearchIndexDict, settlements []*geomodel.Settlement, points []geomodel.Point, seenIds map[int]bool) ([]geomodel.Point, map[int]bool, bool) {

	suitableStation := getSuitableStationFilter(settlements)
	stations, err := findStationsFiltered(pointSearch.Storage, matchType, title, suitableStation)

	if err == nil {
		if matchType == storage.NameSearchExact && pointSearch.Cfg.ReturnSingleCityIfOneCityAndNoOutsideNonBusStations && len(settlements) == 1 {
			settlementID := settlements[0].GetID()
			var outsideNonBusStationsCount int
			for _, s := range stations {
				id, _ := s.GetSettlementID()
				if id != settlementID && s.GetTransportType() != geomodel.ETransportBus {
					outsideNonBusStationsCount += 1
				}
			}
			if outsideNonBusStationsCount == 0 {
				return points, seenIds, true // TODO: check logic with original
			}
		}

		filteredStations := make([]*geomodel.Station, defaultBlockSize)
		for _, p := range stations {
			id := p.GetID()
			_, found := seenIds[id]
			if !found && conformTType(p, tType) {
				seenIds[id] = true
				filteredStations = append(filteredStations, p)
			}
		}
		sortedStations := sortedStations(filteredStations)
		for _, s := range sortedStations {
			points = append(points, geomodel.Point(s))
		}
	}
	return points, seenIds, false
}

// Возвращает список найденных объектов
// Страны, найденные NameSearch Exact
// Города, найденные NameSearch Exact, отсортированные по (majority, id)
// Станции, найденные NameSearch Exact, отсортированные по (majority, title)
// Страны, найденные NameSearch Words
// Города, найденные NameSearch Words и по iata коду, отсортированные по (majority, id)
// Станции, найденные NameSearch Words и code_manager, отсортированные по (majority, title)
func (pointSearch BasePointSearch) GetPointBlocks(title string, tType geomodel.TransportType) ([]geomodel.Point, map[int]bool, error) {
	seenIds := make(map[int]bool, pointSearch.Cfg.SearchPointLimit)
	result := make([]geomodel.Point, pointSearch.Cfg.SearchPointLimit)

	// exact

	result, seenIds = pointSearch.AppendCountries(title, storage.NameSearchExact, result, seenIds)
	result, seenIds, settlements := pointSearch.AppendSettlements(title, tType, storage.NameSearchExact, result, seenIds)
	result, seenIds, needReturn := pointSearch.AppendStations(title, tType, storage.NameSearchExact, settlements, result, seenIds)
	if needReturn {
		return result, seenIds, nil
	}

	// words

	result, seenIds = pointSearch.AppendCountries(title, storage.NameSearchWords, result, seenIds)
	result, seenIds, settlements = pointSearch.AppendSettlements(title, tType, storage.NameSearchWords, result, seenIds)
	result, seenIds, _ = pointSearch.AppendStations(title, tType, storage.NameSearchWords, settlements, result, seenIds)

	// codes // TODO
	//code = title
	//yield 1, Settlement.hidden_manager.get_list(iata__iexact=code)
	//yield 1, Station.code_manager.get_list_by_code(code)

	return result, seenIds, nil
}

func (pointSearch BasePointSearch) PointSearch(title string, tType geomodel.TransportType) ([]geomodel.Point, map[int]bool, error) {
	var result []geomodel.Point

	pointID, pointType, found := pointSearch.Storage.DefaultPointRepo.GetPointIDByTitle(title)
	if found {
		point, err := pointSearch.Storage.GetPointByID(pointID, pointType, true)
		if err == nil && conformTType(point, tType) {
			pointID = point.GetID()
			result = append(result, point)
		}
	}

	points, seenIds, err := pointSearch.GetPointBlocks(title, tType)
	if err != nil {
		return nil, nil, err // TODO may be return one default point
	}
	if found {
		for _, p := range points {
			id := p.GetID()
			if id != pointID {
				result = append(result, p)
			}
		}
		seenIds[pointID] = true
	}

	return result, seenIds, nil
}
